open Ast
open Proof

(* Extracted lambda term *)

type ext =
  | Fun of int * ext     (* lambda *)
  | Raa of int * ext     (* reductio ad absurdum *)
  | Apply of ext * ext   (* function application *)
  | Pair of ext * ext    (* product *)
  | Gen of term * ext    (* generalization *)
  | Fst of ext           (* left projection *)
  | Snd of ext           (* right projection *)
  | Spec of term * ext   (* projection by term *)
  | Unit                 (* unit *)
  | Inl of ext           (* left injection *)
  | Inr of ext           (* right injection *)
  | In of term * ext     (* injection by term *)
  | Out of term * ext *ext (* sum by term *)
  | Match of ext * ext * ext (* sum *)
  | Efq of ext           (* ex falso quodlibet *)
  | Magic                (* excluded middle *)
  | Task of int
  | Var of int

(* Extract the lambda term from a proof *)
let rec extract (p : proof) : ext =
    match p with
  | Iff_intro (Iff (_, _), p1, p2) -> Pair (extract p1, extract p2) 
  | Arrow_intro (Imp (_, _), Assumption (x, p1), p2) -> Fun (x, extract p2) 
  | Arrow_elim (_, p1, p2) -> Apply (extract p2, extract p1)
  | And_intro (And (_, _), p1, p2) -> Pair (extract p1, extract p2)
  | And_elim_left (_, p1) -> Fst (extract p1)
  | And_elim_right (_, p1) -> Snd (extract p1)
  | Or_intro_left (Or (_, _), p1) -> Inl (extract p1)
  | Or_intro_right (Or (_, _), p1) -> Inr (extract p1)
  | Or_elim (_, p1, p2, p3) -> Match (extract p1, extract p2, extract p3)
  | Not_intro (Not _, p1) -> extract p1
  | Not_elim (Imp (_, _), p1) -> extract p1
  | Exists_intro (t, Exists (_, _), p1) -> In (t, extract p1)
  | Exists_elim (t, _, p1, p2) -> Out (t, extract p1, extract p2)
  | Forall_intro (t, Forall (_, _), p1) -> Gen (t, extract p1)
  | Forall_elim (t, _, p1) -> Spec (t, extract p1)
  | Proof.Efq (_, p1) -> Efq (extract p1)
  | Proof.Magic _ -> Magic
  | Proof.Raa (_, Assumption (x, p1), p2) -> Raa (x, extract p2)
  | Proof.Unit -> Unit
  | Assumption (x, _) -> Var x
  | Proof.Task x -> Task x
  | _ -> failwith "Bad proof"

let rec to_string (p : ext) : string =
  match p with
  | Fun (x, p1) -> Printf.sprintf "(fun %s -> %s)" (to_string (Var x)) (to_string p1)
  | Raa (x, p1) -> Printf.sprintf "(fun %s -> %s)" (to_string (Var x)) (to_string p1)
  | Apply (p1, p2) -> Printf.sprintf "(%s %s)" (to_string p1) (to_string p2)
  | Pair (p1, p2) -> Printf.sprintf "(%s, %s)" (to_string p1) (to_string p2)
(*  | Gen (t, p1) -> Printf.sprintf "(Gen %s %s)" (term_to_string t) (to_string p1)*)
  | Gen (t, p1) -> Printf.sprintf "(fun %s -> %s)" (term_to_string t) (to_string p1)
  | Fst p1 -> Printf.sprintf "(fst %s)" (to_string p1)
  | Snd p1 -> Printf.sprintf "(snd %s)" (to_string p1)
(*  | Spec (t, p1) -> Printf.sprintf "(Spec %s %s)" (term_to_string t) (to_string p1)*)
  | Spec (t, p1) -> Printf.sprintf "(%s (%s))" (to_string p1) (term_to_string t)
  | Unit -> "unit"
  | Inl p1 -> Printf.sprintf "(Inl %s)" (to_string p1)
  | Inr p1 -> Printf.sprintf "(Inr %s)" (to_string p1)
  | In (t, p1) -> Printf.sprintf "(In %s %s)" (term_to_string t) (to_string p1)
  | Out (t, p1, p2) -> Printf.sprintf "(Out %s %s %s)" (term_to_string t) (to_string p1) (to_string p2)
  | Match (p1, p2, p3) ->
      let (x, y, z) = (to_string p1, to_string p2, to_string p3) in
      Printf.sprintf "(match %s with Inl y -> (%s y) | Inr z -> (%s z))" x y z
  | Efq p1 -> Printf.sprintf "(efq %s)" (to_string p1)
  | Magic -> "magic"
  | Task x -> "T" ^ string_of_int x
  | Var x -> "x" ^ string_of_int x
