// Project:			Fixed Length Linked List
// Source File:		List.h
// Author:			James Ezick
// Creation Date:	February 27, 1999
//
// Description:
// This class provides the services of a static length linked list.  All
// memory used to hold list information and pointers to data members are 
// allocated at compile time on the stack.  All elements of the list are indexed
// from one (1) with the singular exception of references in array form ([]) where
// elements are indexed from zero (0).  This class does range checking on insertion
// and does not allow the insertion of NULL pointer.
//
// Notes: (Strengths)
// This class allows for linear time iteration and constant time 
// insertion for elements inserted at the head. It is optimized
// for insertion at the head and iteration.
//
// Notes: (Weaknesses)
// This class does NOT have constant time random access.  The numerical list 
// position is not stored with the marker (it is an iteration only tool).  Hence,
// random access requires O(n/2) (linear) time to find an element.  This is unlike
// an array which DOES have constant time random access.  It should be noted that
// constant time random access could be provided at the cost of an additional array
// of size SIZE and a constant time additional work on each of insertion, deletion and
// swapping.
//
// Notes: Begin and end flags
// These two flags are iteration tools.  When an attempt is made to itererate past
// one of the end elements, the relavent flag is set.  This indicated that iteration
// should stop.  The flags are reset when the marker is moved to an interior element.
// The flags are both set to true when the list is empty.
//

#ifndef LIST_H
#define LIST_H

#include "element.h"
#include "utilities.h"
#include <iostream>
using std::ostream;  // Don't include an entire namespace for an included class

#define SIZE 50   // Maximum number of list elements

class List
{
public:
	List();
	List(const List& CopyList, const Integer Deep);
	~List();

	void AddTail(element_t* NewElement);
	void AddHead(element_t* NewElement);
	void Insert(element_t* NewElement);
	void Insert(element_t* NewElement, const UnsignedInteger NewElementListPosition);
	element_t* Update(element_t* NewElement);
	element_t* Update(element_t* NewElement, const UnsignedInteger Position);
	void Swap(const UnsignedInteger SourceIndex);
	void Swap(const UnsignedInteger SourceIndex, const UnsignedInteger DestinationIndex);
	element_t* Delete();
	element_t* Delete(const UnsignedInteger Position);
	void Destroy();
	void Destroy(const UnsignedInteger Position);
	void Remove(const element_t& Element, const Integer Destroy);
	void Sort();
	void Empty(const Integer Destroy);
	
	element_t* Get() const;
	element_t* Get(const UnsignedInteger n) const;
	element_t* GetHead() const;
	element_t* GetTail() const;
	UnsignedInteger GetLength() const;

	Integer First() const;
	Integer Next() const;
	Integer Previous() const;
	Integer Last() const;
	Integer NextRing() const;
	Integer PreviousRing() const;
	Integer Set(const UnsignedInteger n) const;
	
	Boolean IsEmpty() const;
	Boolean IsNotEmpty() const;
	Boolean IsFull() const;
	Boolean IsNotFull() const;
	Boolean IsBeginning() const;
	Boolean IsNotBeginning() const;
	Boolean IsEnd() const;
	Boolean IsNotEnd() const;
	Boolean IsMember(const element_t& Element, const Integer SetPointer) const;
	Boolean IsNotMember(const element_t& Element) const;

	void Print() const;
	void PrintList() const;

	Boolean operator==(const List& TestList) const;
	Boolean operator!=(const List& TestList) const;

	element_t* operator++() const;
	element_t* operator--() const;
	element_t* operator++(Integer) const;
	element_t* operator--(Integer) const;

	element_t*& operator[](const UnsignedInteger Position);
	List& operator=(const List& AssignList);

// Note:  A friend is not a member of a class.  Rather it is a declaration
// of a "friend" of the class.  The problem here is that the left hand operand
// of operator<< is not a List, it is an ostream.  Therfore this cannot be a method
// of class List.  Further, we do not have the right to add this to class ostream.
// We declare it then as a simple function.  Making it a friend means that it can 
// access the private members of class List, even though it is not a member.
friend ostream& operator<<(ostream& OutputStream, const List& OutputList);

private:
	UnsignedInteger next[SIZE];
	element_t* value[SIZE];
	UnsignedInteger previous[SIZE];

	UnsignedInteger available[SIZE];
	UnsignedInteger top;

	UnsignedInteger head;
	UnsignedInteger tail;
	mutable UnsignedInteger marker;
	
	mutable Boolean begin_flag;  // mutable flags can be changed in const methods.
	mutable Boolean end_flag;
	
	enum{ FALSE = 0, TRUE};   // Internal boolean constants
};

inline UnsignedInteger List::GetLength() const
{
	return top;
}

inline Boolean List::IsEmpty() const
{
	return(top==0);
}

inline Boolean List::IsNotEmpty() const
{
	return(top!=0);
}

inline Boolean List::IsFull() const
{
	return(top==SIZE);
}

inline Boolean List::IsNotFull() const
{
	return(top != SIZE);
}

inline Boolean List::IsBeginning() const
{
	return begin_flag;
}

inline Boolean List::IsNotBeginning() const
{
	return !begin_flag;
}

inline Boolean List::IsEnd() const
{
	return end_flag;
}

inline Boolean List::IsNotEnd() const
{
	return !end_flag;
}

#endif
