// Project:			Fixed Length Linked List
// Source File:		List.cpp
// Author:			James Ezick
// Creation Date:	February 27, 1999
//
// Description:
// This class provides the services of a static length linked list.  All
// memory used to hold list information and pointer to data members is 
// allocated at compile time on the stack.  All elements of the list are indexed
// from one (1) with the singular exception of references in array for ([]) where
// elements are indexed from zero (0).  This class does range checking on insertion
// and does not allow the insertion of NULL pointer.
//

#include "list.h"

List::List()
// Defalut constructor, initialize the list to empty.
{
	head = ZERO;
	tail = ZERO;
	top = ZERO;
	marker = ZERO;
	begin_flag = TRUE;
	end_flag = TRUE;  // List is empty
	for(Integer Counter = ZERO; Counter < SIZE; ++Counter)
	{
		value[Counter] = ZERO;
		next[Counter] = ZERO;
		previous[Counter] = ZERO;
		available[Counter] = Counter;
	}
}

List::List(const List& CopyList, const Integer Deep=FALSE)
// Copy constructor with one default parameter.  If Deep is True, the copies of each element
// marker to by a list element is copied.  If Deep is false, then the new list points to 
// the same set of elements.
{
	if(Deep == TRUE)   // Deep copy
	{
		head = ZERO;
		tail = ZERO;
		top = ZERO;
		marker = ZERO;
		begin_flag = TRUE;
		end_flag = TRUE;  // List is empty
		for(Integer Counter = ZERO; Counter < SIZE; ++Counter)
		{
			value[Counter] = ZERO;
			next[Counter] = ZERO;
			previous[Counter] = ZERO;
			available[Counter] = Counter;
		}
		for(UnsignedInteger CopyListIterator = CopyList.head, CopyListCounter = ZERO;
			CopyListCounter < CopyList.GetLength();
			CopyListIterator = CopyList.next[CopyListIterator], ++CopyListCounter)
		{
			AddTail(new element_t(*(CopyList.value[CopyListIterator])));
		}
	}
	else       // Shallow copy (Default)
	{
		head = CopyList.head;
		tail = CopyList.tail;
		marker = CopyList.marker;
		top = CopyList.top;
		begin_flag = CopyList.begin_flag;
		end_flag = CopyList.end_flag;
		for(Integer Counter=ZERO; Counter < SIZE; ++Counter)
		{
			next[Counter] = CopyList.next[Counter];
			value[Counter] = CopyList.value[Counter];
			previous[Counter] = CopyList.previous[Counter];
			available[Counter] = CopyList.available[Counter];
		}
	}
}

List::~List()
// Destructor.  Removes all elements from the list.  Does NOT destroy the elements in the list.
{
	// Check destruction of Arrays
	Empty(FALSE);
}

void List::AddTail(element_t* NewElement)   // Add to tail
// Add an element to the tail (last position) of the list.  Check to see that the list is
// not full and that the element is not a marker to NULL.
{
	if(IsFull() == TRUE)   // List is Full
		ReportError("Cannot add an element to a full list.");
	else if(NewElement == ZERO)   // marker is NULL
		ReportError("Cannot add a NULL marker to the list.");
	else
	{
		UnsignedInteger ListCellForNewElement = available[top];
		value[ListCellForNewElement] = NewElement;

		if(IsEmpty() == TRUE)   // List is Empty
		{
			++top;
			head = ListCellForNewElement;
			previous[ListCellForNewElement] = ListCellForNewElement;
		}
		else
		{
			++top;
			previous[ListCellForNewElement] = tail;
			next[tail] = ListCellForNewElement;
		}
		next[ListCellForNewElement] = ListCellForNewElement;
		tail = ListCellForNewElement;
		marker = ListCellForNewElement;   // Set marker to new element
		begin_flag = FALSE;   // Reset end flags
		end_flag = FALSE;   
	}
}

void List::AddHead(element_t* NewElement)
// Add an element to the head (first position) of the list.  Error checking is done by
// the Insert method.	
{
	Insert(NewElement,ONE);   // Insert the new element (NewElement) into Position ONE (Head).
}

void List::Insert(element_t* NewElement)
// Inserts a new element into the position of the list marker.  Check for a full list and 
// for the new element being a NULL marker.
{
	if(IsFull()==TRUE)
		ReportError("Cannot insert into a full list.");
	else if(NewElement == ZERO)
		ReportError("Cannot insert a NULL marker into the list.");
	else
	{
		if((IsEmpty()==TRUE) || (end_flag==TRUE))   // List is empty or add to tail
			AddTail(NewElement);     // Use Add to tail method
		else
		{
			UnsignedInteger ListCellForNewElement = available[top++];
			value[ListCellForNewElement] = NewElement;

			if(marker == head)    // Add to head
			{
				previous[ListCellForNewElement] = ListCellForNewElement;
				head = ListCellForNewElement;
			}
			else   // Add an interior element at position marker
			{
				previous[ListCellForNewElement] =  previous[marker];
				next[previous[marker]] = ListCellForNewElement;
			}
			next[ListCellForNewElement] = marker;
			previous[marker] = ListCellForNewElement;
			marker = ListCellForNewElement;   // set marker to new element
			begin_flag = FALSE;    // reset end flags
			end_flag = FALSE;
		}
	}
}

void List::Insert(element_t* NewElement, const UnsignedInteger NewElementListPosition)
// Inserts a new element into a list at a position defined by the NewPositon parameter.
// It is assumed that list elements are indexed beginning from one.
{
	if(NewElementListPosition <= ONE)   // Insert to head
	{
		First();
		Insert(NewElement);
		if(NewElementListPosition == ZERO) begin_flag = TRUE;
	}
	else if(NewElementListPosition > GetLength())  // Insert to tail
	{
		AddTail(NewElement);  // Tail insertion
		if(NewElementListPosition > GetLength()) end_flag = TRUE;
	}
	else
	{
		Set(NewElementListPosition);   // Set the list marker to the new position
		Insert(NewElement);   // Insert to the new position
	}
}

element_t* List::Update(element_t* Element)
// Updates the value of the list element at the position of the marker.
{
	if(Element == ZERO)
	{
		ReportError("Cannot update a data member to NULL.");
		return Element;
	}
	else
	{
		element_t* OldElement = value[marker];  // return the element being replaced
		value[marker] = Element;
		return OldElement;
	}
}

element_t* List::Update(element_t* Element, const UnsignedInteger Position)
// Updates the value of the list element at the position defined by the second parameter.
{
	if((Position <= SIZE) && (Position != ZERO))
	{
		Set(Position);
		return Update(Element);
	}
	else
	{
		std::cerr << "List position is out of range." << std::endl;
		return ZERO;
	}
}

void List::Swap(const UnsignedInteger SourceIndex)
// Swap exchanges the value of the list element at the source index and the list element at
// the list marker.  If the elements are the same the method has no effect.
{
	if(SourceIndex == ZERO)
		ReportError("List elements are indexed from one (1).");
	else if(SourceIndex > GetLength())
		ReportError("One or both index parameters are larger than the size of the list.");
	else
	{
		UnsignedInteger old_marker = marker;   // Save the current marker position
		element_t*& Source = (*this)[SourceIndex-1];
		element_t* Temp = Source;
		Source = value[old_marker];
		value[old_marker] = Temp;
		marker = old_marker;   // Restrore the marker position
	}
}

void List::Swap(const UnsignedInteger SourceIndex, const UnsignedInteger DestinationIndex)
// Swaps the list element at the SourceIndex with the list element at the Destination Index.
{
	if(SourceIndex == ZERO)
		ReportError("List elements are indexed from one (1).");
	else if((SourceIndex > GetLength()) || (DestinationIndex > GetLength()))
		ReportError("One or both index parameters are larger than the size of the list.");
	else if(SourceIndex == DestinationIndex);
		// Do nothing.
	else
	{
		UnsignedInteger old_marker = marker;
		Set(DestinationIndex);
		Swap(SourceIndex);
		marker = old_marker;
	}
}

element_t* List::Delete()
// Removes from the list the element at the current marker position.
{	
	if(IsEmpty() == TRUE)   // List is empty
	{
		ReportError("Cannot delete an element from an empty list");
		return ZERO;
	}
	else
	{
		element_t* DeletedElement = Get();
		available[--top] = marker;   // Restore location to available list	
		next[previous[marker]] = marker[next];
		previous[next[marker]] = marker[previous];
		if(marker == tail)        // Tail removed
			marker = tail = previous[marker];
		else if(marker == head)   // Head removed
			marker = head = next[marker];
		else
			marker = next[marker];
		begin_flag = IsEmpty();
		end_flag = IsEmpty();  // Set both flags if list empty
		return DeletedElement;
	}
}

element_t* List::Delete(const UnsignedInteger Position)
// Removed from the list the element at the specified position.
{
	if(Position > top)
	{
		ReportError("The position you wish to delete is out of range.");
		return ZERO;
	}
	else
	{
		Set(Position);
		return(Delete());
	}
}

void List::Destroy()
// Removes from the list the element at the current position and calls its destuctor.
{	
	if(IsEmpty() == TRUE)   // List is empty
		ReportError("Cannot delete an element from an empty list");
	else
	{
		element_t* DestroyedElement = Delete();
		DestroyedElement->~element_t();
		delete DestroyedElement;
	}
}

void List::Destroy(const UnsignedInteger Position)
// Removes from the list the element at the specified position and calls its destuctor.
{
	if(Position > top)
		ReportError("The position you wish to delete is out of range.");
	else
	{
		Set(Position);
		Destroy();
	}
}

void List::Remove(const element_t& Element, const Integer Destroy=FALSE)
// Deletes all occurances of a element in the list.
{
	while(IsMember(Element,TRUE))   // Delete all occurances
		Delete(Destroy);
}

void List::Empty(const Integer DestroyFlag=FALSE)
{
	if(DestroyFlag==TRUE)
		while(IsNotEmpty()==TRUE) Destroy();
	else
		while(IsNotEmpty()==TRUE) Delete();
	begin_flag = TRUE;
	end_flag = TRUE;
}

void List::Sort()
// Sort the list into increasing order using an insertion sort. (O(n^2) sort)
{
	for(UnsignedInteger ComparisonListCounter = ZERO, ComparisonElementIndex = head;
		ComparisonListCounter < (GetLength()-1);
		++ComparisonListCounter, ComparisonElementIndex = next[ComparisonElementIndex])
	{
		UnsignedInteger SmallestRemainingElementIndex = ComparisonElementIndex;
		for(UnsignedInteger SwapListCounter = ComparisonListCounter+1,
			SwapElementIndex = next[ComparisonElementIndex];
			SwapListCounter < GetLength();
			++SwapListCounter, SwapElementIndex = next[SwapElementIndex])
		{
			if(*(value[SwapElementIndex]) < (*(value[SmallestRemainingElementIndex])))
			{
				SmallestRemainingElementIndex = SwapElementIndex;
			}
		}
		if(ComparisonElementIndex != SmallestRemainingElementIndex)  // Swap the elements
		{
			element_t* temp = value[ComparisonElementIndex];
			value[ComparisonElementIndex] = value[SmallestRemainingElementIndex];
			value[SmallestRemainingElementIndex] = temp;
		}
	}
	First();   // Reset the marker to first element
}

element_t* List::Get() const
// Return the list element at the current marker position.
{
	if(IsNotEmpty())
		return value[marker];
	else
	{
		// ReportError("Cannot retrieve an element from an empty list.");
		return ZERO; // Not we may which to not report an error and check from NULL return.
	}
}

element_t* List::Get(const UnsignedInteger Position) const
// Return the list element at the specified position.
{
	if(Position > GetLength())
	{
		ReportError("You have attempted to retrieve a member which is out of range.");
		return ZERO;
	}
	else
	{
		Set(Position);
		return Get();
	}
}

element_t* List::GetHead() const
// Set the marker to the first element and return it.
{
	Set(ONE);
	return Get();
}

element_t* List::GetTail() const
// Set the marker to the last element and return it.
{
	Set(GetLength());
	return Get();
}

Integer List::First() const
// Set the marker to the first element.
{
	if(IsEmpty() == TRUE)   // List is empty
		return FALSE;
	else
	{
		Set(ONE);
		return TRUE;
	}
}

Integer List::Next() const
{
	if(IsNotEmpty() == TRUE)   // List not empty
		begin_flag = FALSE;   // reset the beginning flag
	if(marker==tail)   // Go past the end
	{
		end_flag = TRUE;
		return FALSE;
	}
	else
	{
		marker = next[marker];
		return TRUE;
	}	
}

Integer List::Previous() const
{
	if(IsNotEmpty() == TRUE)   // List not empty 
		end_flag = FALSE;   // reset the end flag
	if(marker == head)  // Go past beginning
	{
		begin_flag = TRUE;
		return FALSE;
	}
	else
	{
		marker = previous[marker];
		return TRUE;
	}
}

Integer List::Last() const
{
	if(IsEmpty() == TRUE)
		return FALSE;
	else
	{
		Set(GetLength());
		return TRUE;
	}
}

Integer List::NextRing() const
{
	if(IsEmpty() == TRUE)
		return FALSE;
	else if(marker == tail)
	{
		marker = head;
		end_flag = FALSE;
		return TRUE;
	}
	else
	{
		marker = next[marker];
		return TRUE;
	}
}

Integer List::PreviousRing() const
{
	if(IsEmpty() == TRUE)
		return FALSE;
	else if(marker == head)
	{	
		marker = tail;
		begin_flag = FALSE;
		return TRUE;
	}
	else
	{
		marker = previous[marker];
		return TRUE;
	}
}

Integer List::Set(const UnsignedInteger Position) const
// Set the position of the list marker to a specified position (indexing the list from one).
{
	if(IsEmpty()==TRUE)   // List is empty
	{
		// ReportError("Cannot set the position within an empty list.");
		return FALSE;
	}
	else if(Position == 0)  // Point before the beginning
	{
		begin_flag = TRUE;
		end_flag = FALSE;
		marker = head;
		return TRUE;
	}
	else if(Position == GetLength()+1)  // Point past the end
	{
		begin_flag = FALSE;
		end_flag = FALSE;
		marker = tail;
		return TRUE;
	}
	else if(Position > GetLength()+1)   // Position is out of range
	{
		ReportError("You have attempted to set a position beyond the range of the list.");
		return FALSE;
	}
	else
	{
		if(Position < ((GetLength()+1)/2))   // Begin itterating from the beginning
		{
			marker = head;
			for(UnsignedInteger Counter=1;Counter < Position; ++Counter) 
				marker = next[marker];
		}
		else            // Begin itterating from the end
		{
			marker = tail;
			for(UnsignedInteger ReverseCounter=GetLength(); ReverseCounter > Position; --ReverseCounter) 
				marker = previous[marker];
		}
		begin_flag = FALSE;
		end_flag = FALSE;  // marker now points to an interior item
		return TRUE;
	}
}		

Boolean List::IsMember(const element_t& Element, const Integer SetMarker=FALSE) const
// Return TRUE if the element is in the list, FALSE if it is not.  If Set marker is true
// the the marker is set to the position of the first found element.
{
	Boolean FOUND = FALSE;
	UnsignedInteger OldPointer = marker;
	
	marker = head;
	for(UnsignedInteger Iterator=0; (Iterator < GetLength()) && (FOUND==FALSE); ++Iterator)
	{
		FOUND = (*(value[marker]) == Element);
		marker = next[marker];
	}
	if(!FOUND || !SetMarker)
		marker = OldPointer;   // Restore marker location
	return FOUND;
}

Boolean List::IsNotMember(const element_t& Element) const
{
	return !IsMember(Element,FALSE);
}

void List::Print() const
{
	if(IsNotEmpty())
		std::cout << *(value[marker]) << " ";
	else
		std::cout << "Empty ";
}

void List::PrintList() const
{
	UnsignedInteger old_marker = marker;
	marker = head;
	for(UnsignedInteger Counter=0; Counter<GetLength(); ++Counter)
	{
		std::cout << *(value[marker]) << " ";
		marker = next[marker];
	}
	std::cout << std::endl;
	marker = old_marker;   // Restore old marker value
}

Boolean List::operator==(const List& TestList) const
{
	Integer EQUAL = (top == TestList.top);   // Same length
	
	UnsignedInteger old_marker = marker;
	UnsignedInteger l_old_pointer = TestList.marker;
	
	marker = head;
	TestList.marker = TestList.head;
	for(UnsignedInteger Iterator=0;((Iterator < GetLength()) && (EQUAL==TRUE)); ++Iterator)
	{
		EQUAL = (*(TestList.value[TestList.marker]) == *(value[marker]));
		marker = next[marker];
		TestList.marker = next[TestList.marker];
	}	
	marker = old_marker;
	TestList.marker = l_old_pointer;
	return(EQUAL==TRUE);
}

Boolean List::operator!=(const List& TestList) const
{
	return !(*this==TestList);
}

element_t* List::operator++() const
{
	Next();
	return Get();
}

element_t* List::operator--() const
{
	Previous();
	return Get();
}

element_t* List::operator++(Integer) const
{
	element_t* Temp = Get();
	Next();
	return Temp;
}

element_t* List::operator--(Integer) const
{
	element_t* Temp = Get();
	Previous();
	return Temp;
}

element_t*& List::operator[](const UnsignedInteger Position)
// Array style indexing, first element is denoted as postion zero [0].
{
	if(Position < GetLength())
	{
		if(Position < (GetLength() / 2))
		{
			for(UnsignedInteger Place=head,Counter=0; Counter<Position; ++Counter)
				Place = next[Place];
			return value[Place];
		}
		else
		{
			for(UnsignedInteger Place=tail,Counter=top-1; Counter>Position; --Counter)
				Place = previous[Place];
			return value[Place];
		}
	}
	else
	{
		ReportError("You have attempted to retrieve an element beyond the range of the list.");
		return value[0];   // this is a dummy return reference
	}
}

List& List::operator=(const List& AssignList)   // Shallow copy
// Assignement operator that checks for self assignment and then performs a shallow copy.
// Note that the check for equality takes linear time.
{
	if(&AssignList != this)
	{
		head = AssignList.head;
		tail = AssignList.tail;
		marker = AssignList.marker;
		top = AssignList.top;
		begin_flag = AssignList.begin_flag;
		end_flag = AssignList.end_flag;
		for(Integer Counter=ZERO; Counter < SIZE; ++Counter)
		{
			next[Counter] = AssignList.next[Counter];
			value[Counter] = AssignList.value[Counter];
			previous[Counter] = AssignList.previous[Counter];
			available[Counter] = AssignList.available[Counter];
		}
	}
	return *this;
}

ostream& operator<<(ostream& OutputStream, const List& OutputList)
{
	UnsignedInteger old_marker = OutputList.marker;
	OutputList.marker = OutputList.head;
	for(UnsignedInteger Counter=ZERO; Counter < OutputList.GetLength(); ++Counter)
	{
		OutputStream << *(OutputList.value[OutputList.marker]) << " ";
		OutputList.marker = OutputList.next[OutputList.marker];
	}
	OutputStream << std::endl;
	OutputList.marker = old_marker;
	return OutputStream;
}
