/**
 *  File: tileDemo.cpp
 *  Probject: Tile Demo
 *  Last Update: 11/29/07
 *  Description: a demo program on tile background.
 */

#include "gba/gba.h"
#include "graphics.h"

int main() {

 /* In this demo, we show how to initialize and set the tiles
  * on tiled background (mode 0, background 0), and a simple
  * example on scrolling.
  * The files included in this demo are graphics.h/.cpp, which
  * hold the graphics data of tiles.
  *
  * Setting the mode: mode 0, map 1d, background 0.
  */
  SetMode(MODE_0 | OBJ_ENABLE | OBJ_MAP_1D | BG0_ENABLE);

 /* Initialize:
  * A tiled background consists of a chunk of 8x8 tiles. The
  * size of the chunk depends on the background mode and its
  * attributes. Here we show how to initialize a 256x256 tile
  * background for background 0 (in mode 0).
  *
  * First we set the background 0 register with the following
  * flags: color 256, size 256x256, screen base block 0, and
  * char base block 8.
  */
  REG_BG0CNT = BG_COLOR_256 | TEXTBG_SIZE_256x256 |
                    (8 << SCREEN_SHIFT) | (0 << CHAR_SHIFT);

 /* Like sprites, tiled background needs to refer to an
  * array of tiles to draw the tiles on screen.
  * So we first copy the graphics data in graphics.cpp
  * to the video memory - character base block. The expression
  * REG_BG0CNT = 0 << CHAR_SHIFT specifies which character
  * base block to use.
  *
  * First we create a pointer to the CharBaseBlock, and then
  * copy the data to the CharBaseBlock array.
  */
  u16* cbb = (u16*)CharBaseBlock(0);
  for (int i=0; i<TILE_DATA_LENGTH; i++)
  {
   cbb[i] = tileData[i];
  }

 /* Like sprites, we also need to specify the palette for
  * the tiles:
  */
  for (int i=0; i<TILE_PALETTE_LENGTH; i++)
  {
   BG_PaletteMem[i] = tilePalette[i];
  }

 /* To put tiles on screen, we tell the gba to look into
  * an area in video memory called ScreenBaseBlock that
  * contain the information on how the tiles are laid out.
  * The expression REG_BG0CNT = 8 << SCREEN_SHIFT specifies
  * which screen base block to use.
  * The screen base block stores the 2-D background as a 1-D
  * array. For text background size 256x256 (the current
  * setting), screen base block stores one row of 256 pixel
  * at a time. So to change the first tile on the second row,
  * we will have to access the 33th element in the screen
  * base block. (We have to create a pointer to SBB first)
  */
  u16* sbb = (u16*)ScreenBaseBlock(8);
  // set all to tile 0.
  for (int i=0; i<32*32; i++)
  {
    sbb[i] = 0;
  }
  // set first tile on second row to tile 1.
  sbb[32] = 1;
  // set more tiles to 1.
  for (int i=15; i<32*32; i+=32)
  {
    sbb[i] = 1;
  }

 /* Note that for text backgrounds with size larger than 256x256,
  * The SBB stores a portion of the background at a time. i.e. if
  * the background is 512x256, it will store the left half
  * (0,0) - (31,31), and then the right half (32,0) - (63,31)
  * So if we wanted to change the 33th tile on the first row,
  * we have to access the (32*32+1)-th element in SBB.
  *
  * Scrolling
  * "Scrolling the background" means to move the screen over the
  * background so it will display things that were originally
  * out of sight. The scrolling is controlled by the vertical/
  * horizontal offset registers for the particular background.
  * In our case, they are REG_BG0VOFS and REG_BG0HOFS defined in
  * gba_bg.h. The offsets represent the position of the top-left
  * point of the screen in the background.
  * Simply compile ane run this program to see the effects of
  * the following code:
  */
  u16 h = 0;
  while (true)
  {
    if (keyDown(KEY_RIGHT))
      h++;
    else if (keyDown(KEY_LEFT))
      h--;
    REG_BG0HOFS = h;
    vsync();
  }

  return 0;
}
