package simple;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

/**
 * A scanner for the Simple language.
 *
 * For CS 212, Feb 2008.
 *
 * Spaces, tabs, and end-of-line are ignored. Comments start with "#" and go to
 * the end of the line.  The valid tokens are
 *   word = a letter followed by 0 or more letters and/or digits,
 *   number = one or more digits, and
 *   character = any single character that is not a letter or digit.
 *
 * @author Paul Chew
 */
public class SimpleScanner extends AbstractList<String> {

    private List<String> tokenList = new ArrayList<String>();
    private int nextIndex = 0;

    /**
     * @param input a character source (e.g., a FileReader or a StringReader)
     * @throws IOException
     */
    public SimpleScanner (Reader input) throws IOException {
        char state;            // Current state (x=nothing, w=word, n=number)
        StringBuffer token;    // Current token
        token = new StringBuffer();
        BufferedReader br = new BufferedReader(input);
        for (String line = br.readLine(); line != null; line = br.readLine()) {
            state = 'x';
            for (int i = 0; i < line.length(); i++) {
                char ch = line.charAt(i);
                if (ch == '#') {
                    i = line.length();      // Set so we skip rest of line
                } else if (state == 'x') {
                    if (Character.isLetter(ch)) {
                        token = new StringBuffer("" + ch);
                        state = 'w';
                    } else if (Character.isDigit(ch)) {
                        token = new StringBuffer("" + ch);
                        state = 'n';
                    } else if (Character.isWhitespace(ch)) {
                        state = 'x';
                    }
                    else {
                        tokenList.add(""+ ch);
                    }
                } else if (state == 'w') {
                    if (Character.isLetterOrDigit(ch)) {
                        token.append(ch);
                    } else {
                        tokenList.add(token.toString());
                        state = 'x';
                        i--;        // Set so we look at ch again
                    }
                } else if (state == 'n') {
                    if (Character.isDigit(ch)) {
                        token.append(ch);
                    } else {
                        tokenList.add(token.toString());
                        state = 'x';
                        i--;        // Set so we look at ch again
                    }
                }
            }
            // Handle any word or number that we had started
            if (state == 'w' || state == 'n') {
                tokenList.add(token.toString());
            }
        }
    }

    /**
     * @return true iff there are more tokens
     */
    public boolean hasNext () {
        return nextIndex < tokenList.size();
    }

    /**
     * @param token the token to look for
     * @return true iff the specified String appears as the next token
     */
    public boolean hasNext (String token) {
        return token.equals(tokenList.get(nextIndex));
    }

    /**
     * @return true iff the next token is an integer
     */
    public boolean hasNextInt () {
        return (hasNext() &&
                Character.isDigit(tokenList.get(nextIndex).charAt(0)));
    }

    /**
     * @return true iff the next token is a word
     */
    public boolean hasNextWord () {
        return (hasNext() &&
                Character.isLetter(tokenList.get(nextIndex).charAt(0)));
    }

    /**
     * @return the next token
     */
    public String next () {
        return tokenList.get(nextIndex++);
    }

    /**
     * @return the next token which must be an integer
     * @throws NumberFormatException if the next token is not a valid integer
     */
    public int nextInt () throws NumberFormatException {
        return new Integer(tokenList.get(nextIndex++));
    }

    @Override
    public String get(int index) {
        return tokenList.get(index);
    }

    @Override
    public int size() {
        return tokenList.size();
    }
    /**
     * Main program; used for testing.
     */
    public static void main (String[] args) throws IOException {
        Reader input = new StringReader("do 5:x=x +191");
        SimpleScanner scanner = new SimpleScanner(input);
        for (String token : scanner) System.out.println(token);
    }

}
