package simple;

import java.util.ArrayList;
import java.util.List;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.FileWriter;

/**
 * An example parser for the Simple language.
 *
 * For CS 212, Feb 2008.
 *
 * The grammar for the Simple language is shown below.
 * '*' indicates 0 or more occurrences; '|' indicates choice; an item within
 * brackets [] is optional; parentheses () are used for grouping.
 * '*' and parentheses also appears as a token of the language, but
 * this usage should be clear from context.
 *   program -> statement* end .
 *   statement -> name = expression ;
 *   statement -> do expression : statement* end ;
 *   expression -> part [(+|-|*|/) part]
 *   part -> name | number | (expression)
 *   name -> <single letter a-z>
 *
 * @author Paul Chew
 */
public class Parser {

    SimpleScanner scanner;              // The scanner

    /**
     * @param scanner the scanner to be used during parsing
     */
    public Parser (SimpleScanner scanner) {
        this.scanner = scanner;
    }

    /**
     * Test program.
     */
    public static void main (String[] args) throws IOException {
        // Prepare a file for input
        String fileName = "test.txt";
        if (args.length == 1) fileName = args[0];
        FileReader f = new FileReader(fileName);
        // Prepare the scanner
        SimpleScanner scanner = new SimpleScanner(f);
        // Build the AST and display it
        Ast tree = (new Parser(scanner)).parseProgram();
        System.out.println(tree);
        // Generate code
        PrintWriter out = new PrintWriter(new FileWriter("test.sam"));
        tree.generate(out);
        // Clean up
        f.close();  out.close();
    }

    /**
     * Consume a specific token.
     * @param token the token to consume
     * @throws ParseError if the next token is not the one expected
     */
    public void eat (String token) throws ParseError {
        String next = scanner.next();
        if (next.equals(token)) return;
        throw new ParseError("Expected " + token + ", not " + next);
    }

    /**
     * Scan a letter token.
     * @return the letter found
     * @throws ParseError if the next token is not a letter
     */
    public char scanLetter () throws ParseError {
        String next = scanner.next();
        if (next.length() == 1 && 'a' < next.charAt(0) && next.charAt(0) < 'z')
            return next.charAt(0);
        throw new ParseError("Expected a letter, not " + next);
    }


    /**
     * Parse a program.
     * @return the AST (Abstract Syntax Tree) for the program
     */
    public Ast parseProgram () {
        List<Ast> statements = new ArrayList<Ast>();
        while (!scanner.hasNext("end")) {
            statements.add(parseStatement());
        }
        eat("end"); eat(".");
        if (scanner.hasNext())
            throw new ParseError("Extra tokens after end of program");
        return new AstProgram(statements);
    }

    public Ast parseStatement () {
        if (scanner.hasNext("do")) {            // Do statement
            eat("do");
            Ast expression = parseExpression();
            eat(":");
            List<Ast> statements = new ArrayList<Ast>();
            while (!scanner.hasNext("end")) {
                statements.add(parseStatement());
            }
            eat("end"); eat(";");
            return new AstDo(expression, statements);
        } else {                                // Assignment statement
            char target = scanLetter();
            eat("=");
            Ast expression = parseExpression();
            eat(";");
            return new AstAssignment(target, expression);
        }
    }

    public Ast parseExpression () {
        Ast left = parsePart();
        String token;
        if (scanner.hasNext("+") || scanner.hasNext("-") ||
                scanner.hasNext("*") || scanner.hasNext("/")) {
            token = scanner.next();
            Ast right = parsePart();
            return new AstOperator(token.charAt(0), left, right);
        } else {
            return left;
        }
    }

    public Ast parsePart () {
        if (scanner.hasNextInt()) {             // Integer
            int value = scanner.nextInt();
            return new AstNumber(value);
        } else if (scanner.hasNext("\\(")) {    // Parenthesized expression
            eat("(");
            Ast expression = parseExpression();
            eat(")");
            return expression;
        } else {                                // Variable
            return new AstVariable(scanLetter());
        }
    }
}

/**
 * Parsing error.
 */
class ParseError extends RuntimeException {
    public ParseError (String string) {
        super(string);
    }
}

