/**
 *  File: sprite.h
 *  Project: 212SP08 Assignment 2
 *  Description: declaration of Sprite object
 *
 */

#ifndef SPRITE_H
#define SPRITE_H

#include "gba/gba.h"
#include "defs.h"
#include "util.h"
#include "graphics.h"

/** A Sprite represents a sprite object. It handles sprite
 *  operations. It is a (very) simple version of sprite manager.
 *  Some options (e.g. rotation, layer) are omitted to suit the
 *  the purpose of this assignment.
 *  The position, velocity, acceleration, etc of the sprite
 *  use FIXED POINT MATH. The resolution FIXPT is defined in defs.h.
 */

 // definitions
#define MAX_SPRITE_NUM  128 /* maximum number of sprite */

class Sprite {

  public:

    /** static members */

    /** Redraw all sprite objects and wait for vsync.
     *  use copyOamLimited() defined in util.h.
     *
     * Arguments
     *  none
     *
     * Returns
     *  none
     */
    static void draw();

    /** Construct a sprite with following default attributes:
     *  rotation : off
     *  double size : off
     *  alpha-blending : none
     *  mosaic : off
     *  rotation data index : 0
     *  priority : 0
     *  direction : east
     *  velocity : 0
     *  acceleration : 0
     *  speed cap : -1
     *  animation : no
     *
     *  The newly instantiated will point to the next available slot in
     *  OBJ_Mem. If there are already MAX_SPRITE_NUM sprites in memory,
     *  this constructor will do nothing.
     *
     * Arguments
     *  u16 firstFrame : the index of OAM_Data for the first frame
     *  u16 colorDepth : color depth of sprite. vals defined in gba.h
     *  u16 palette : the palette number (for 16 colors only)
     *  u16 size : size of sprite. use vals defined in gba.h
     *  u16 shape : size of sprite. use vals defined in gba.h
     *  s16 x : initial x-coordinate on screen (IN FIXED POINT)
     *  s16 y : initial y-coordinate on screen (IN FIXED POINT)
     */
    Sprite (u16 firstFrame, u16 colorDepth, u16 palette,
                u16 size, u16 shape, s16 x, s16 y);

    /** Deconstruct the sprite.
     */
    ~Sprite();

    /** Move the sprite in the direction indicated.
     *  This move will be influenced by the velocity,
     *  acceleration, and speed cap of the sprite.
     *
     * Arguments
     *  none
     *
     * Returns
     *  none
     */
    void move();

    /** Determine whether this sprite collides with the other
     *  sprite in question. Collision is defined as the rectangle on
     *  screen of one sprite overlap the other.
     *
     * Arguments
     *  Sprite *other : the other sprite in question.
     *
     * Returns
     *  bool : true if they collide.
     */
    bool isCollidedWith(Sprite *other);

    /** OPTIONAL Set the animation parameters
     *  Example: for an forward animation of a size_16, color_256
     *           sprite with first frame = 1 and cycle length = 2;
     *           the frames of the animation will be 1, 9, 1, 9, etc.
     *           for backward, first frame = 1, cycle length = 3;
     *           the frames of the animation will be 1, 17, 9, 1, 17, etc.
     *
     * Arguments
     *  u16 firstFrame : the first frame of the animation cycle
     *                   (the index to the OJB_Data)
     *  u16 cycleDir : the direction of the animation cycle
     *  u16 cycleLength : the length of the animation
     *                    (how many frames in the cycle?)
     */
    void setAnimation(u16 firstFrame, u16 cycleDir, u16 cycleLength);

    /** OPTIONAL Forward one frame in the animation cycle direction
     *  see specs for setAnimation for more details
     *
     * Aruments
     *  none
     *
     * Returns
     *  none
     */
    void animate();

    /** Setters and Getters
     *  The following functions are setters and getters for the
     *  sprite attributes.
     *  Note that some attributes are not in this list as they
     *  are intended to be handled by Sprite class only.
     */

    /** palette: the palette number of the sprite.
     */
    void setPalette(u16 palette);
    /** shape: the shape of the sprite (defined in gba_sprites.h)
     *         can be SQUARE, WIDE, TALL;
     */
    void setShape(u16 shape);
    /** size: the size of the sprite (defined in gba_sprites.h)
     *         can be SIZE_8, SIZE_16, SIZE_32, SIZE_64.
     */
    void setSize(u16 size);
    /** direction: the way the sprite is facing
     *             enum Direction defined in defs.h.
     */
    Direction getDirection();
    void setDirection(Direction direction);
    /** color depth: (defined in gba_sprites.h)
     *                can be COLOR_16, COLOR_256.
     */
    void setColorDepth(u16 colorDepth);
    /** horizontal flip: true if sprite is horizontally flipped.
     */
    bool isHFlip();
    void setHFlip(bool hFlip);
    /** vertical flip: true if sprite is vertically flipped.
     */
    bool isVFlip();
    void setVFlip(bool vFlip);
    /** position: the position of the sprite (IN FIXED POINT)
     */
    Point getPosition();
    void setPosition(s16 x, s16 y);
    /** velocity: the velocity of the sprite (IN FIXED POINT)
     */
    Point getVelocity();
    void setVelocity(s16 x, s16 y);
    /** acceleration: the acceleration of the sprite (IN FIXED POINT)
     */
    Point getAcceleration();
    void setAcceleration(s16 x, s16 y);
    /** speed cap: the max speed of the sprite (IN FIXED POINT)
     *             for effect of friction. set to -1 if no limit.
     */
    Point getSpeedCap();
    void setSpeedCap(s16 x, s16 y);
    /** dimension: returns the dimension (in pixels) of the sprite
     *             use function in gbaUtil.
     */
    Point getDimension();
    /** current frame: the current frame of the sprite. it is the index
     *                 of the OBJ_Data which determines what graphics
     *                 the sprite is using.
     */
    void setCurrentFrame(u16 currentFrame);
    /** OPTIONAL
     *  cycle direction: which way in OBJ_Data the sprite animation frames
     *                   are changing. 1 = forward, -1 = backward,
     *                   0 = no animation.
     */
    void setCycleDir(s16 cycleDir);

  private:

    /** static members */
    static OAMEntry* shadow_; /* sprite memory shadow for local update.
                                 copied to OAM_Data in draw(). */

    /** Initialize the sprites. Copy data to OAM_Data.
     *  Must be called only once and only before any object
     *  is instantiated.
     */
    static void initialize();

    /** constants */
    const static u16 xMask = 0x01FF;   /* bit field mask for x-coordinate */
    const static u16 yMask = 0x00FF;   /* bit field mask for y-coordinate */
    const static u16 pMask = 0xF000;   /* bit field mask for palette */

    /** Commit the changes to the sprite attributes to the
     *  local OAMEntry shadow.
     *
     * Arugments
     *  none
     *
     * Returns
     *  none
     */
    void commit();

    //
    // MORE PRIVATE MEMBERS HERE AS NEEDED
    //

};

#endif
