import java.io.PrintWriter;

/**
 * This represents an expression of the form (expr op expr)
 * If the second expression is null it represents (op expr)
 */
public class ComplexExpression implements Expression {
	private Operator oper = Operator.NOT;
	private Expression value1 = null;
	private Expression value2 = null;

	public static enum Operator{
		/**
		 * Boolean NOT
		 */
		NOT,

		/**
		 * Integer negation
		 */
		NEGATE,

		/**
		 * Addition of two integers
		 */
		ADD,

		/**
		 * Subtraction of two integers
		 */
		SUBTRACT,

		/**
		 * Mutliplication of two integers
		 */
		MULTIPLY,

		/**
		 * Division of two integers
		 */
		DIVIDE,

		/**
		 * Modulus of two integers
		 */
		MODULUS,
	
		/**
		 * Equality of two expressions
		 */
		EQUALITY,
        
		/**
		 * Inequality of two expressions
		 */
		INEQUALITY,

		/**
		 * Comparison of two integers (n1 &lt; n2)
		 */
		LESSTHAN,
        
		/**
		 * Comparison of two integers (n1 &lt;= n2)
		 */
		LESSTHANOREQUAL,

		/**
		 * Comparison of two integers (n1 &gt; n2)
		 */
		GREATERTHAN,

		/**
		 * Comparison of two integers (n1 &gt;= n2)
		 */
		GREATERTHANOREQUAL,

		/**
		 * Boolean OR (short-circuiting)
		 */
		OR,

		/**
		 * Boolean AND (short-circuiting)
		 */
		AND,

		/**
		 * Boolean XOR
		 */
		XOR,

		/**
          * Assignment operator
          */
        ASSIGNMENT,
	}

	/**
	 * Creates a ComplexExpression 
	 */
	public ComplexExpression() { }

	/**
	 * Sets the first expression of this expression
	 * @param v1 The first expression in this expression
	 */
	public void setValue1(Expression v1) {
		value1 = v1;
	}

	/**
	 * Sets the second expression of this expression
	 * @param v2 The second expression in this expression
	 */
	public void setValue2(Expression v2) {
		value2 = v2;
	}

	/**
	 * Sets the operator for this expression 
	 * @param op The operator for this expression
	 */
	public void setOperator(Operator op) {
		oper = op;
	}

	/**
	 * Prints the SaM code for this expression
	 * @param out The print writer
	 */
	public void printSamCode(PrintWriter out) {
		int number = Utils.getNumber();

		switch (oper) {

			case NOT :
				value1.printSamCode(out);
				out.println("NOT");
				break;

			case NEGATE :
				out.println("PUSHIMM 0");
				value1.printSamCode(out);
				out.println("SUB");
				break;

			case ADD :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("ADD");
				break;

			case SUBTRACT :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("SUB");
				break;

			case MULTIPLY :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("TIMES");
				break;

			case DIVIDE :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("DIV");
				break;

			case MODULUS :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("MOD");
				break;
					
			case EQUALITY :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("EQUAL");
				break;

			case INEQUALITY :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("EQUAL");
				out.println("NOT");
				break;

			case LESSTHAN :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("CMP");
				out.println("ISPOS");
				break;

			case LESSTHANOREQUAL :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("CMP");
				out.println("ISNEG");
				out.println("NOT");
				break;

			case GREATERTHAN :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("CMP");
				out.println("ISNEG");
				break;

			case GREATERTHANOREQUAL :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("CMP");
				out.println("ISPOS");
				out.println("NOT");
				break;

			case OR :
				value1.printSamCode(out);
				/* Short circuit
				 * We need to duplicate it,
				 * since JUMPC will eat the old one */
				out.println("DUP");
				/* Now Jump if this is true */
				out.println("JUMPC end" + number);
				value2.printSamCode(out);
				out.println("OR");
		                out.println("end" + number + ":");

				break;

			case AND :
				value1.printSamCode(out);
				/* Duplicate this and negate it 
				 * (because we need to jump if its false) */
				out.println("DUP");
				out.println("NOT");
				out.println("JUMPC end" + number);
				value2.printSamCode(out);
				out.println("AND");
		                out.println("end" + number + ":");
				break;

			case XOR :
				value1.printSamCode(out);
				value2.printSamCode(out);
				out.println("XOR");
				break;

			case ASSIGNMENT :
				value2.printSamCode(out);
				out.println("DUP");	
				((BaliLvalueNode) value1).printSamLvalueCode(out);				
				break;
		}
	}

	/**
	 * Returns the return type of the expression
	 * @return The data type for this expression 
	 */
	public DataType getReturnType() {

		DataType t1 = value1.getReturnType();
		
		switch (oper) {
			case ASSIGNMENT :
				return t1;
			case ADD :
			case SUBTRACT :
				return DataType.INT;
			case MULTIPLY :
			case DIVIDE :
			case MODULUS :
			case NEGATE :
				return DataType.INT;
			case EQUALITY :
			case INEQUALITY :
			case NOT :
			case LESSTHAN :
			case LESSTHANOREQUAL :
			case GREATERTHAN :
			case GREATERTHANOREQUAL :
			case OR :
			case AND :
			case XOR :
				return DataType.BOOLEAN;

			default:
				return DataType.INT;
		}
	}

}
