import edu.cornell.cs.bali.compiler.BaliSyntaxException;
import edu.cornell.cs.bali.compiler.CS212Compiler;
import edu.cornell.cs.bali.compiler.IllegalBaliException;
import edu.cornell.cs.sam.io.Tokenizer;

import java.io.PrintWriter;
import java.io.StringWriter;

/**
 * This is a Bali to SaM compiler. We first loop though
 * the input file to generate an AST using the BaliNode classes
 * Then the BaliNode classes print out the SAM code
 */
public class BaliCompiler implements CS212Compiler {
	private SymbolTable globals = new SymbolTable(VariableExpression.VariableType.GLOBAL);
	private SymbolTable locals = new SymbolTable(VariableExpression.VariableType.LOCAL);
	private BlockStatement body;
	
	/**
	 * Starts the compiler
	 * @param in The Bali program 
	 * @return The compiled program as a string
	 */
	public String compile(Tokenizer in) throws IllegalBaliException {
		StringWriter outs = new StringWriter();
		PrintWriter out = new PrintWriter(outs);
		
		processGlobalVariables(in);
		processMain(in);
	
        BaliProgram prog = new BaliProgram(globals, locals, body);
		prog.printSamCode(out);

		return outs.toString();
	}

	/**
	 * Adds global variables to the global variable symbol table
	 */
	private void processGlobalVariables(Tokenizer in) 
		throws IllegalBaliException{

		/* Global variables are optional */
		if(!in.check("global")) return;
		
		if (!in.check('{')) 
			throw new BaliSyntaxException("Expected '{' in front of global variable block", in.nextLineNo());

		/* Loop through and process declarations */
		while (!in.check('}')) 
			processVarDecl(in, globals);
	}

	/**
	 * Adds local variables to the local variable symbol table
	 */
	private void processLocalVariables(Tokenizer in) throws IllegalBaliException{

		if (!in.check('{')) 
			throw new BaliSyntaxException("Expected '{' in front of local variable block", in.nextLineNo());

		/* Loop through and process declarations */
		while (!in.check('}')) 
			processVarDecl(in, locals);
	}
	
	/**
	 * Processes a variable declaration and adds it to a symbol table
 	 */
	private void processVarDecl(Tokenizer in, SymbolTable table) 
		throws BaliSyntaxException {

		DataType type = processType(in);
		do {
			if (in.peekAtKind() != Tokenizer.TokenType.WORD)
				throw new BaliSyntaxException("Expected variable name", in.nextLineNo());

			table.addVariable(in.getWord(), type);
		}
		while (in.check(','));

		if (!in.check(';'))
			throw new BaliSyntaxException("Expected ; after variable declaration", in.nextLineNo());
	}	
	
	/**
	 * Processes the function header and returns a Function
	 */
	private void processMain(Tokenizer in) throws IllegalBaliException {
		if (in.peekAtKind() != Tokenizer.TokenType.WORD)
			throw new BaliSyntaxException("Expected type", in.nextLineNo());
		
		/* No semantic error checking was done for this part. 
		 * Assumed next token is "int".
		 */
		in.getWord();

		if (in.peekAtKind() != Tokenizer.TokenType.WORD)
			throw new BaliSyntaxException("Expected function name", in.nextLineNo());

		/* No semantic error checking was done for this part. 
		 * Assume next token is "main".
		 */

		in.getWord();

		if (!in.check('('))
			throw new BaliSyntaxException("Expected ( after function name", in.nextLineNo());

		if (!in.check(')'))
			throw new BaliSyntaxException("Expected ) after function parameters", in.nextLineNo());

		processLocalVariables(in);
		body = processBlockStatement(in);
	}
	
	/**
	 * Processes a type
	 */
	private DataType processType(Tokenizer in) throws BaliSyntaxException {
		if (in.peekAtKind() != Tokenizer.TokenType.WORD)
			throw new BaliSyntaxException("Expected type", in.nextLineNo());
		String type = in.getWord();
		
		/* Parse base type */
		DataType base = null;
		if (type.equals("int")) 
			base = DataType.INT;
		else if (type.equals("boolean")) 
			base = DataType.BOOLEAN;
		else if (type.equals("char")) 
			base = DataType.CHAR;
		else
			throw new BaliSyntaxException("Invalid type", in.lineNo());	
		
		return base;
	}

	/**
	 * Processes a statement. There are several different types
	 * of statements - if/else, print, do...until, for, return and an expression
	 */
	private Statement processStatement(Tokenizer in) throws IllegalBaliException {

		if (in.test("if"))
			return processIfStatement(in);
		else if (in.test("do"))
			return processDoUntilStatement(in);
		else if (in.test("print"))
			return processPrintStatement(in);
		else if (in.test("return"))
			return processReturnStatement(in);
		else if (in.test("for"))
			return processForStatement(in);
		else if (in.test('{'))
			return processBlockStatement(in);
		else if (in.check(';'))
			return new EmptyStatement();

		/* Expression Statement */
		else {
			Expression expr1 = processExpression(in);
			if (in.check(';')) {
				ExpressionStatement es = new ExpressionStatement();
				es.setExpression(expr1);
				return es;
			} 
			else throw new BaliSyntaxException("Expected ; after expresison statement", 
				in.nextLineNo());
		}
	}

	/**
	 * Returns a new block statement
	 */
	private BlockStatement processBlockStatement(Tokenizer in) 
		throws IllegalBaliException {

		if (!in.check('{'))
			throw new BaliSyntaxException("Expected {", in.nextLineNo());
		BlockStatement stat = new BlockStatement();
		while (!in.check('}')) {
			stat.addStatement(processStatement(in));
		}
		return stat;
	}
	
	/**
	 * Processes an if statement and the block(s) that follow it
	 */
	private Statement processIfStatement(Tokenizer in) 
		throws IllegalBaliException {

		if (!in.check("if"))
			throw new BaliSyntaxException("Expected if", in.nextLineNo());

                IfStatement s = new IfStatement();

		if (!in.check('('))
			throw new BaliSyntaxException("Expected '(' before " +
				"the condition expression", in.nextLineNo());

		s.setIfExpr(processExpression(in));

                if (!in.check(')'))
                        throw new BaliSyntaxException("Expected ')' after " + 
				" the condition expression", in.nextLineNo());

		s.setThenStatement(processStatement(in));

		if (in.check("else")) 
			s.setElseStatement(processStatement(in));

		return s;
	}

	/**
	 * Process a for statement and the block following it
	 */
	private Statement processForStatement(Tokenizer in) throws IllegalBaliException {

		if (!in.check("for"))
			throw new BaliSyntaxException("Expected for", in.nextLineNo());
		
		ForStatement f = new ForStatement();
		
		if (!in.check('('))
			throw new BaliSyntaxException("Expected '(' after for", in.nextLineNo());
	
		if (!in.check(';')) {
			f.setInitExpr(processExpression(in));	
			if (!in.check(';'))
				throw new BaliSyntaxException("Expected ';' after initial loop expression", in.nextLineNo());
		}

		if (!in.test(';')) 
			f.setCondition(processExpression(in));
		else
			f.setCondition(new ConstantExpression(true));
			
		if (!in.check(';'))
			throw new BaliSyntaxException("Expected ';' after loop condition", in.nextLineNo());
	
		if (!in.test(')'))
			f.setRepeatExpr(processExpression(in));

		if (!in.check(')'))
			throw new BaliSyntaxException("Expected ')' after loop", in.nextLineNo());
		
		f.setStatement(processStatement(in));
		return f;
	}		

	/**
	 * Processes a DoUntil statement and the block following it
	 */
	private Statement processDoUntilStatement(Tokenizer in) throws IllegalBaliException {

		if (!in.check("do"))
			throw new BaliSyntaxException("Expected do", in.nextLineNo());

		DoUntilStatement s = new DoUntilStatement();
		s.setLoopStatement(processStatement(in));

		if (!in.check("until")) 
			throw new BaliSyntaxException("Expected 'until'", in.nextLineNo());

                if (!in.check('(')) 
                        throw new BaliSyntaxException("Expected '(' before " + 
				"the condition expression", in.nextLineNo());

		s.setCondition(processExpression(in));

                if (!in.check(')')) 
                        throw new BaliSyntaxException("Expected ')' after " +
				"the condition expression", in.nextLineNo());

		if (!in.check(';'))
			throw new BaliSyntaxException("Expected ; after loop statement", in.nextLineNo());

		return s;
	}

	/**
	 * Processes a print statement
	 */
	private Statement processPrintStatement(Tokenizer in) throws IllegalBaliException {
		
		if (!in.check("print"))
			throw new BaliSyntaxException("Expected print", in.lineNo());

		PrintStatement s = new PrintStatement();
		s.setExpression(processExpression(in));

		if (!in.check(';'))
			throw new BaliSyntaxException("Expected ; after print statement", in.nextLineNo());

		return s;
	}

	/**
	 * Process a return statement
	 */
	private Statement processReturnStatement(Tokenizer in) throws IllegalBaliException {
		int returnLineNo = in.nextLineNo();

		if (!in.check("return"))
			throw new BaliSyntaxException("Expected return", returnLineNo);

		ReturnStatement s = new ReturnStatement(Utils.mainEndCallString);
		s.setExpression(processExpression(in));

		if (!in.check(';'))
			throw new BaliSyntaxException("Expected ; after return statement", in.nextLineNo());

		return s;
	}

	/**
	 * This processes an expression, which can be any number of things
	 */
	private Expression processExpression(Tokenizer in) throws IllegalBaliException {

		/* Process part 1 */
		Expression part1 = processExpressionPart(in);

		/* See if we now have an operator */
		if (in.peekAtKind() != Tokenizer.TokenType.OPERATOR) 
			return part1;

		/* Check Operator */
		ComplexExpression.Operator op;
		switch (in.getOp()) {
			case '+' :
				op = ComplexExpression.Operator.ADD;
				break;
			case '-' :
				op = ComplexExpression.Operator.SUBTRACT;
				break;
			case '*' :
				op = ComplexExpression.Operator.MULTIPLY;
				break;
			case '/' :
				op = ComplexExpression.Operator.DIVIDE;
				break;
			case '%' :
				op = ComplexExpression.Operator.MODULUS;
				break;

			case '>' :
				if (in.check('='))
					op = ComplexExpression.Operator.GREATERTHANOREQUAL;
				else
					op = ComplexExpression.Operator.GREATERTHAN;
				break;

			case '<' :
				if (in.check('='))
					op = ComplexExpression.Operator.LESSTHANOREQUAL;
				else
					op = ComplexExpression.Operator.LESSTHAN;
				break;

			case '=' :
				if (!in.check('=')) 
					op = ComplexExpression.Operator.ASSIGNMENT;
				else 
					op = ComplexExpression.Operator.EQUALITY;
				break;

			case '!' :
				if (!in.check('='))
					throw new BaliSyntaxException("Expected = after !", in.nextLineNo());
				op = ComplexExpression.Operator.INEQUALITY;
				break;

			case '|' :
				if (!in.check('|'))
					throw new BaliSyntaxException("Expected | after |", in.nextLineNo());
				op = ComplexExpression.Operator.OR;
				break;

			case '&' :
				if (!in.check('&'))
					throw new BaliSyntaxException("Expected & after &", in.nextLineNo());
				op = ComplexExpression.Operator.AND;
				break;

			case '^' :
				op = ComplexExpression.Operator.XOR;
				break;

			default :
				in.pushBack();
				return part1;
		}

		/* Now we need another expression part for the binary expression */
		ComplexExpression e = new ComplexExpression();
		e.setValue1(part1);
		e.setOperator(op);

		if (op == ComplexExpression.Operator.ASSIGNMENT) 
			e.setValue2(processExpression(in));
		else
			e.setValue2(processExpressionPart(in));
			
		return e;
	}

	/**
	 * Processes an Expression Part
	 */
	private Expression processExpressionPart(Tokenizer in) throws IllegalBaliException {

		Expression epart = null;

		/* Check for integer constant */
		if (in.peekAtKind() == Tokenizer.TokenType.INTEGER) 
			epart = new ConstantExpression(in.getInt());
		
		/* Check for character constant */
		else if (in.peekAtKind() == Tokenizer.TokenType.CHARACTER) {
			epart = new ConstantExpression(in.getCharacter());
		}
		
		/* Check for boolean constant */
		else if (in.check("true"))
			epart = new ConstantExpression(true);
		else if (in.check("false"))
			epart = new ConstantExpression(false);

	
		/* Check for nested expression */
		else if (in.check('(')) {
			Expression e = processExpression(in);
			if(!in.check(')')) throw new BaliSyntaxException("Expected )", in.nextLineNo());
			epart = e;
		}

		/* Check for unary expression */
		else if (in.check('-')) {
			ComplexExpression e = new ComplexExpression();
			e.setValue1(processExpressionPart(in));
			e.setOperator(ComplexExpression.Operator.NEGATE);
			epart = e;
		}
		else if (in.check('!')) {
			ComplexExpression e = new ComplexExpression();
			e.setValue1(processExpressionPart(in));
			e.setOperator(ComplexExpression.Operator.NOT);
			epart = e;
		}

		/* Builtin functions */
		else if (in.check("readInt")) {
			if (!in.check('(') || !in.check(')'))
				throw new BaliSyntaxException("Expected ( ) after readInt call", in.nextLineNo());
			epart = new ReadIntExpression();
		}

		else if (in.check("readChar")) {
		
			if (!in.check('(') || !in.check(')'))
				throw new BaliSyntaxException("Expected ( ) after readChar call", in.nextLineNo());
			epart = new ReadCharExpression();
		}
			
		/* Variable */
		else if (in.peekAtKind() == Tokenizer.TokenType.WORD){
			String word = in.getWord();
			epart = locals.getVariable(word);
			if (epart == null)
				epart = globals.getVariable(word);
		}

		/* Otherwise, it's invalid */
		else 
			throw new BaliSyntaxException("Invalid Expression", in.nextLineNo());

		return epart;
	}
}
