package edu.cornell.cs.sam.io;

/* Tokenizer interface. */

public interface Tokenizer {
	
	public enum TokenType 
		{ INTEGER, FLOAT, WORD, OPERATOR, COMMENT, CHARACTER, STRING, EOF; }
		
	public enum TokenizerOptions
		{ PROCESS_COMMENTS, PROCESS_STRINGS, PROCESS_CHARACTERS; }	

	/* Token Definitions:
	 *
	 * INT: a sequence of digits with only
	 * digits after the first digit. An integer will
	 * always be >= 0.
	 *
	 * FLOAT: a sequence of digits starting with either a
	 * period, or a digit and containing only
	 * digits and one period after the first digit.
	 * A float will always be >= 0.
	 *
	 * WORD: a letter followed by a sequence of alphanumeric characters
	 * or underscores (_) without any whitespace. If a string such as 2.0.0 is detected,
	 * this should also be considered a word.
	 * 
	 * STRING: If string processing is turned on, a
	 * word is anything between two quotation signs. Valid
	 * escapes (\n, \r. \\, \". etc...) should also be evaluated
	 * in this case.
	 *
	 * CHARACTER: a single character between two apostrophes. Valid
	 * escapes are allowed
	 *
	 * OPERATOR: any non-alphanumeric character
	 *
	 * COMMENT: the text following // on a single line.
	 *
	 * EOF: the end of file 
	 */
		
	/**
	 * Returns the type of the next token
	 */
	TokenType peekAtKind() throws TokenizerException;
	/**
	 * Returns the next token (unless its not an integer)
	 * @throws TokenizerException if the next token is not an integer
	 */
	int getInt() throws TokenizerException;
	/**
	 * Returns the next token (unless its not a float)
	 * @throws TokenizerException if the next token is not a float
	 */
	float getFloat() throws TokenizerException;
	/**
	 * Returns the next token (unless its not a word)
	 * @throws TokenizerException if the next token is not a word
	 */
	String getWord() throws TokenizerException;
	/**
	 * Returns the next token (unless its not a string)
	 * @throws TokenizerException if the next token is not a string
	 */
	String getString() throws TokenizerException;
	/**
	 * Returns the next token (unless its not a character)
	 * @throws TokenizerException if the next token is not a character
	 */
	char getCharacter() throws TokenizerException;	/**
	 * Returns the next token (unless its not an operator)
	 * @throws TokenizerException if the next token is not an operator
	 */
	char getOp() throws TokenizerException;
	/**
	 * Returns the next token (unless its not a comment)
	 * @throws TokenizerException if the next token is not a comment
	 */
	String getComment() throws TokenizerException;
	/**
	 * Matches the next character if its a character or throws TokenizerException
	 * @param c the character to match
	 * @throws TokenizerException if the next token is not c
	 */
	void match(char c) throws TokenizerException;
	/**
	 * Matches the next character if its a word or throws TokenizerException
	 * @param s the word to match
	 * @throws TokenizerException if the next token is not s
	 */
	void match(String s) throws TokenizerException;
	/**
	 * Checks if the next token is a character/operator and is c.
	 * If it is, the token is eaten, otherwise it remains in the input
	 * stack.
	 * @param c the value to check against
	 * @return true if the next token is c, false otherwise
	 */
	boolean check(char c);
	/**
	 * Checks if the next token is the word s.
	 * If it is, the token is eaten, otherwise it remains in the input
	 * stack.
	 * @param s the value to check against
	 * @return true if the next token is s, false otherwise
	 */
	boolean check(String s);
	/**
	 * Checks if the next token is a character/operator and is c.
	 * The token is kept regardless of whether it is c or not.
	 * @param c the value to check against
	 * @return true if the next token is c, false otherwise
	 */
	boolean test(char c);
	/**
	 * Checks if the next token is a word and is s. The token is kept
	 * regardless of whether it is s or not
	 * @param s the value to check against
	 * @return true if the next token is s, false otherwise
	 */
	boolean test(String s);
	/**
	 * Pushes the latest token requested back. Implementations
	 * are required to be able to push back to the start of the file.
	 */
	void pushBack();
	/**
	 * Checks if the stream can be pushed back.
	 * @return true if it can be pushed back, false if at the beginning of the file
	 */
	boolean canPushBack();
	/**
	 * Returns the whitespace in the file before the current token. This may be null.
	 */
	String getWhitespaceBeforeToken();
	/**
	 * Returns the line number
	 */
	int lineNo();
	/**
	 * Closes the tokenizer
	 * @throws TokenizerException if there is an I/O Error
	 */
	void close() throws TokenizerException;
}
