package edu.cornell.cs.sam.io;

import java.io.*;
import java.util.*;

/**
 * This is a SamTokenizer implementation that
 * has full backwards/forwards mobility and loads everything into
 * memory as soon as it is created. By default, quote processing
 * is disabled and comment ignoring is enabled. You may use
 * one of the constructors with options to enabled/disable this options
 */
public class SamTokenizer implements Tokenizer {
	/**
	 * The source
	 */
	protected PushbackReader in;
	/**
	 * The unread tokens
	 */
	protected Stack<Token> tokens = new Stack<Token> ();
	/**
	 * The read tokens
	 */
	protected Stack<Token> readTokens = new Stack<Token> ();
	/**
	 * String Processing setting
	 */
	protected boolean processStrings = false;
	/**
	 * Character Processing setting
	 */
	protected boolean processCharacters = false;
	/**
	 * Comment Processing setting
	 */
	protected boolean processComments = false;

	/**
	 * Creates a new SamTokenizer with a file for a source
	 * @param FileName The file name of the file to read
	 * @param opt Options for parsing
	 * @throws IOException If there is a file error
	 * @throws FileNotFoundException If the file could not be found
	 */
	public SamTokenizer(String FileName, Tokenizer.TokenizerOptions... opt)
			throws IOException, FileNotFoundException {
		FileReader r = new FileReader(FileName);
		parseOptions(opt);
		in = new PushbackReader(r);
		read();
	}

	/**
	 * Creates a new SamTokenizer with a Reader as a source
	 * @param r The source
	 * @param opt Options for parsing
	 * @throws IOException if there is a stream error
	 */
	public SamTokenizer(Reader r, Tokenizer.TokenizerOptions... opt) throws IOException {
		in = new PushbackReader(r);
		parseOptions(opt);
		read();
	}

	/**
	 * Creates a new SamTokenizer with System.in as a source
	 * @param opt Options for parsing
	 * @throws IOException if there is a stream error
	 */
	public SamTokenizer(Tokenizer.TokenizerOptions... opt) throws IOException {
		in = new PushbackReader(new InputStreamReader(System.in));
		parseOptions(opt);
		read();
	}
	
	// Parses the options and puts them into action
	private void parseOptions(Tokenizer.TokenizerOptions... opt){
		for(int i = 0; i < opt.length; i++){
			switch(opt[i]){
				case PROCESS_COMMENTS:
					processComments = true;
					break;
				case PROCESS_STRINGS:
					processStrings = true;
					break;
				case PROCESS_CHARACTERS:
					processCharacters = true;
					break;
			}
		}
	}

	// Reads and parses the stream into tokens
	private void read() throws IOException {
		int lineNo = 1;
		int cin;
		char c;
		String whitespace = "";
		for (; (cin = in.read()) != -1; whitespace = "") {
			c = (char)cin;
			if (c == '\n')
				lineNo++;
			// Now we need to detect if its a word, operator, or integer
			if (Character.isWhitespace(c)){
				whitespace += c;
				continue;
			}
			// Skip out the comments
			else if (c == '/') {
				int din = in.read();
				if (din == -1){
					tokens.push(new OperatorToken('/', whitespace, lineNo));
				}
				// This is a valid character but not a comment
				else if ((char)din != '/') {
					in.unread(din);
					tokens.push(new OperatorToken('/', whitespace, lineNo));
				}
				else {
					// OK, so this is a comment
					String comment = "";
					while ((din = in.read()) != -1){
						comment += din;
						if (din == '\n'){
							in.unread(din);
							break;
						}
					}
					if(processComments)
						tokens.push(new CommentToken(comment, whitespace, lineNo));
				}
			}
			// Strings are anything between two " marks
			else if (processStrings && c == '"') {
				String str = "";
				while ((cin = in.read()) != -1 && cin != '"')
					str += readChar(cin, lineNo);
				if(cin == -1) throw new IOException("Missing '\"' on line " + lineNo);
				tokens.push(new StringToken(str, whitespace, lineNo));
			}
			// Characters are like strings except
			else if (processCharacters && c == '\'') {
				cin = in.read();
				if(cin == -1) throw new IOException("Missing ''' on line " + lineNo);
				char a = readChar(cin, lineNo);
				if(in.read() != '\'')
					throw new IOException("Missing ''' on line " + lineNo);
				tokens.push(new CharToken(a, whitespace, lineNo));
			}
			// now it's either an integer or a word or an operator
			else if (Character.isLetter(c)) {
				// It's a word
				String word = "";
				word += c;
				int din;
				while ((din = in.read()) != -1) {
					if (Character.isLetter((char)din) || Character.isDigit((char)din) || din == '_')
						word += (char)din;
					else {
						in.unread(din);
						break;
					}
				}
				tokens.push(new WordToken(word, whitespace, lineNo));
			}
			else if (Character.isDigit(c)) {
				in.unread(c);
				readNumber(lineNo, whitespace, null);
			}
			else {
				// Now this is an operator. The only problem . can also be a number
				// So check for that
				if (c == '.') {
					int din = in.read();
					if (din == -1)
						tokens.push(new OperatorToken(c, whitespace, lineNo));
					else if (Character.isDigit((char)din)) {
						in.unread(din);
						readNumber(lineNo, whitespace, ".");
					}
					else {
						in.unread(din);
						tokens.push(new OperatorToken(c, whitespace, lineNo));
					}
				}
				else
					tokens.push(new OperatorToken(c, whitespace, lineNo));
			}
		}
		// push the EOF token onto the stack
		tokens.push(new EOFToken(whitespace, lineNo));
		// Reverse the stack
		Stack<Token> a = new Stack<Token> ();
		while (!tokens.empty())
			a.push(tokens.pop());
		tokens = a;
	}

	// Reads in a number
	// tok is the start of the number
	private void readNumber(int lineNo, String whitespace, String tok) throws IOException {
		TokenType type = TokenType.INTEGER;
		if (tok == null)
			tok = "";
		else if (tok.equals("."))
			type = TokenType.FLOAT;
		int din;
		while ((din = in.read()) != -1) {
			if (din == '.') {
				switch (type) {
					case INTEGER :
						type = TokenType.FLOAT;
						break;
					case FLOAT :
						type = TokenType.WORD;
						break;
				}
				tok += (char)din;
			}
			else if (Character.isDigit((char)din) || (type == TokenType.WORD 
				&& (Character.isLetter((char)din) || din == '_')))
				tok += (char)din;
			else {
				in.unread(din);
				break;
			}
		}
		try {
			switch (type) {
				case FLOAT :
					tokens.push(new FloatToken(Float.parseFloat(tok), whitespace, lineNo));
					break;
				case INTEGER :
					tokens.push(new IntToken(Integer.parseInt(tok), whitespace, lineNo));
					break;
				default :
					tokens.push(new WordToken(tok, whitespace, lineNo));
			}
		}
		catch (NumberFormatException e) {
			throw new IOException("Error parsing number on line " + lineNo);
		}
	}
	
	// Reads in a character and returns it, parsing escapes if necessary
	private char readChar(int cin, int lineNo) throws IOException{
		if (cin == '\\') {
			cin = in.read();
			if(Character.isDigit(cin)){
				String codeS = "" + (char)cin;
				while(Character.isDigit(cin = in.read())){
					codeS += (char) cin;
					if(codeS.length() > 3)
						throw new IOException("Invalid escape character at " + lineNo);
				}
				in.unread(cin);
				int code;
				try{ code = Integer.parseInt(codeS); }
				catch(NumberFormatException e){
					throw new IOException("Unparsable number at line " + lineNo);
				}
				if(code < 0 || code > 255)
					throw new IOException("Invalid escape character at " + lineNo);
				return (char)code;
			}
			else{
				switch (cin) {
					case 't' :
						return '\t';
					case '\\' :
						return '\\';
					case 'n' :
						return '\n';
					case 'r' :
						return '\r';
					case '"' :
						return '"';
					case '\'':
						return '\'';
					default :
						throw new IOException("Invalid escape character at " + lineNo);
				}
			}
		}
		else
			return (char)cin;
	}


	/**
	 * Return the type of the next token
	 * @return the type of the next token
	 */
	public TokenType peekAtKind() {
		if (tokens.empty())
			return TokenType.EOF;
		return tokens.peek().getType();
	}

	/**
	 * Returns the next token if its an integer
	 */
	public int getInt() throws TokenizerException {
		if (peekAtKind() == TokenType.INTEGER) {
			IntToken i = (IntToken)tokens.pop();
			readTokens.push(i);
			return i.getInt();
		}
		else
			throw new TokenizerException("Attempt to read non-integer value as an integer", lineNo());
	}

	/**
	 * Returns the next token if its a float
	 */
	public float getFloat() throws TokenizerException {
		if (peekAtKind() == TokenType.FLOAT) {
			FloatToken f = (FloatToken)tokens.pop();
			readTokens.push(f);
			return f.getFloat();
		}
		else
			throw new TokenizerException("Attempt to read non-float value as a float", lineNo());
	}

	/**
	 * Returns the next token if its a word
	 */
	public String getWord() throws TokenizerException {
		if (peekAtKind() == TokenType.WORD) {
			WordToken word = (WordToken)tokens.pop();
			readTokens.push(word);
			return word.getWord();
		}
		else
			throw new TokenizerException("Attempt to read non-word value as a word.", lineNo());
	}

	/**
	 * Returns the next token if its a string
	 */
	public String getString() throws TokenizerException {
		if (peekAtKind() == TokenType.STRING) {
			StringToken str = (StringToken)tokens.pop();
			readTokens.push(str);
			return str.getString();
		}
		else
			throw new TokenizerException("Attempt to read non-string value as a string.", lineNo());
	}
	
	/**
	 * Returns the next token if its a character
	 */
	public char getCharacter() throws TokenizerException {
		if (peekAtKind() == TokenType.CHARACTER) {
			CharToken c = (CharToken)tokens.pop();
			readTokens.push(c);
			return c.getChar();
		}
		else
			throw new TokenizerException("Attempt to read non-char value as a char.", lineNo());
	}
	
	/**
	 * Return the next token if its an operator
	 */
	public char getOp() throws TokenizerException {
		if (peekAtKind() == TokenType.OPERATOR) {
			OperatorToken op = (OperatorToken)tokens.pop();
			readTokens.push(op);
			return op.getOp();
		}
		else
			throw new TokenizerException("Attempt to read non-operator value as an op", lineNo());
	}

	/**
	 * Returns the next token if its a comment
	 */
	public String getComment() throws TokenizerException {
		if (peekAtKind() == TokenType.COMMENT) {
			CommentToken c = (CommentToken)tokens.pop();
			readTokens.push(c);
			return c.getComment();
		}
		else
			throw new TokenizerException("Attempt to read non-comment value as a comment", lineNo());
	}

	/**
	 * Matches the next operator or throws TokenizerException
	 * @param c the character to match
	 * @throws TokenizerException if the next token is not c
	 */
	public void match(char c) throws TokenizerException {
		char n;
		if (peekAtKind() == TokenType.OPERATOR) {
			n = getOp();
			if (n != c)
				throw new TokenizerException("Expecting " + c + " but found " + n, lineNo());
		}
		else
			throw new TokenizerException("Did not find " + c, lineNo());
	}

	/**
	 * Matches the next word or throws TokenizerException
	 * @param s the word to match
	 * @throws TokenizerException if the next token is not s
	 */
	public void match(String s) throws TokenizerException {
		String n;
		if (peekAtKind() == TokenType.WORD) {
			n = getWord();
			if (!n.equals(s))
				throw new TokenizerException("Expecting " + s + " but found " + n, lineNo());
		}
		else
			throw new TokenizerException("Did not find " + s, lineNo());
	}

	/**
	 * Checks to see if the next token is an operator and is the provided test case
	 * If it is, it is eaten up, otherwise, it is pushed back 
	 * @param c the operator to check against
	 * @return true if the next token is an operator and is c, false otherwise
	 */
	public boolean check(char c) {
		if (peekAtKind() == TokenType.OPERATOR) {
			if (c == getOp())
				return true;
			else {
				pushBack();
				return false;
			}
		}
		return false;
	}

	/**
	 * Checks to see if the next token is a word and is the provided test case
	 * If it is, it is eaten up, otherwise, it is pushed back
	 * @param s the word to check against
	 * @return true if the next token is a word and is s, false otherwise
	 */
	public boolean check(String s) {
		if (peekAtKind() == TokenType.WORD) {
			if (s.equals(getWord()))
				return true;
			else {
				pushBack();
				return false;
			}
		}
		return false;
	}

	/**
	 * Checks if the next token is a character/operator and is c.
	 * The token is kept regardless of whether it is c or not.
	 * @param c the value to check against
	 * @return true if the next token is c, false otherwise
	 */
	public boolean test(char c) {
		boolean check = check(c);
		if (check)
			pushBack();
		return check;
	}

	/**
	 * Checks if the next token is a word and is s. The token is kept
	 * regardless of whether it is s or not
	 * @param s the value to check against
	 * @return true if the next token is s, false otherwise
	 */
	public boolean test(String s) {
		boolean check = check(s);
		if (check)
			pushBack();
		return check;
	}

	/**
	 * Moves the position of the parser back by one
	 */
	public void pushBack() {
		if (!readTokens.empty())
			tokens.push(readTokens.pop());
	}

	/**
	 * Returns true if there are readTokens
	 */
	public boolean canPushBack() {
		return !readTokens.empty();
	}
	
	/**
	 * Returns the whitespace in the file before the current token.
	 */
	public String getWhitespaceBeforeToken(){
		return tokens.pop().getWhitespace();
	}

	/**
	 * Returns the line number of the last token read
	 * @return the line number of the last token read
	 */
	public int lineNo() {
		if (readTokens.empty())
			return 1;
		else
			return readTokens.peek().lineNo();
	}

	/**
	 * Closes the stream and clears the stored tokens
	 */
	public void close() {
		try {
			tokens.empty();
			readTokens.empty();
			in.close();
		}
		catch (IOException e) { }
	}

	/**
	 * Skips the next token
	 */
	public void skipToken() {
		if (!tokens.empty())
			readTokens.push(tokens.pop());
	}

	/**
	 * This is the parent class for all tokens.
	 */
	static abstract class Token {
		/**
		 * The line number of the token
		 */
		protected int lineNo;
		
		/**
		 * The whitespace that is between the token before
		 * this token and this token.
		 */
		protected String whitespace;
		
		/**
		 * Returns the line number of this token
		 */
		public int lineNo(){
			return lineNo;
		}
		
		/**
		 * Returns the whitespace before this token
		 */
		public String getWhitespace(){
			return whitespace;
		}
		
		/**
		 * Returns type of this token
		 */
		public abstract TokenType getType();
	}

	static class IntToken extends Token {
		int integer;

		public IntToken(int integer, String whitespace, int lineNo) {
			this.lineNo = lineNo;
			this.whitespace = whitespace;
			this.integer = integer;
		}

		public int getInt() {
			return integer;
		}

		public String toString() {
			return whitespace + integer;
		}

		public TokenType getType() {
			return TokenType.INTEGER;
		}
	}

	static class FloatToken extends Token {
		float f;

		public FloatToken(float fl, String whitespace, int lineNo) {
			this.lineNo = lineNo;
			this.whitespace = whitespace;
			this.f = fl;
		}

		public float getFloat() {
			return f;
		}

		public String toString() {
			return whitespace + f;
		}

		public TokenType getType() {
			return TokenType.FLOAT;
		}
	}

	static class WordToken extends Token {
		String word;

		public WordToken(String word, String whitespace, int lineNo) {
			this.lineNo = lineNo;
			this.whitespace = whitespace;
			this.word = word;
		}

		public String getWord() {
			return word;
		}

		public String toString() {
			return whitespace + word;
		}

		public TokenType getType() {
			return TokenType.WORD;
		}
	}

	static class OperatorToken extends Token {
		char op;

		public OperatorToken(char op, String whitespace, int lineNo) {
			this.op = op;
			this.whitespace = whitespace;
			this.lineNo = lineNo;
		}

		public char getOp() {
			return op;
		}

		public String toString() {
			return whitespace + op;
		}

		public TokenType getType() {
			return TokenType.OPERATOR;
		}
	}
	
	static class CommentToken extends Token {
		String comment;
		
		public CommentToken(String comment, String whitespace, int lineNo){
			this.comment = comment;
			this.whitespace = whitespace;
			this.lineNo = lineNo;
		}
		
		public String getComment() {
			return comment;
		}
		
		public String toString() {
			return whitespace + "//" + comment;
		}
		
		public TokenType getType() {
			return TokenType.COMMENT;
		}
	}
	
	static class StringToken extends Token {
		String str;
		
		public StringToken(String str, String whitespace, int lineNo){
			this.str = str;
			this.whitespace = whitespace;
			this.lineNo = lineNo;
		}
		
		public String getString() {
			return str;
		}
		
		public String toString() {
			return whitespace + '"' + str + '"';
		}
		
		public TokenType getType() {
			return TokenType.STRING;
		}
	}

	static class CharToken extends Token {
		char c;
		
		public CharToken(char c, String whitespace, int lineNo){
			this.c = c;
			this.whitespace = whitespace;
			this.lineNo = lineNo;
		}
		
		public char getChar() {
			return c;
		}
		
		public String toString() {
			return whitespace + '\'' + c + '\'';
		}
		
		public TokenType getType() {
			return TokenType.CHARACTER;
		}
	}

	static class EOFToken extends Token {
		public EOFToken(String whitespace, int lineNo){
			this.whitespace = whitespace;
			this.lineNo = lineNo;
		}
		
		public String toString() {
			return whitespace;
		}
		
		public TokenType getType() {
			return TokenType.EOF;
		}
	}
}
