import java.net.*;
import java.io.*;
import java.util.Vector;

public class ServerThread extends Thread {
  private Socket socket;
  private ElectionServer gui;
  private ElectionsInfo electionsInfo;

  public ServerThread(Socket s, ElectionServer serverGUI, ElectionsInfo info) {
    socket = s;
    gui = serverGUI;
    electionsInfo = info;
  }

  //Override the run() method of the Thread class to process Message objects
  //that we get from the client
  public void run() {
    if(socket == null) return;
    //Streams for reading and writing objects through the stream
    ObjectInputStream in = null;
    ObjectOutputStream out = null;

    try {
      //Get object streams for reading/writing
      out = new ObjectOutputStream(socket.getOutputStream());
      in = new ObjectInputStream(socket.getInputStream());

      Object obj;
      //Keep reading in objects
      while ( (obj = in.readObject()) != null) {
        //Skip this object if it is not a Message object
        if (! (obj instanceof Message))
          continue;

        Message msg = (Message) obj;
        //Exit if the message is a disconnect message.  Close all streams and
        //the socket
        if (msg.getType() == Message.DISCONNECT) {
          in.close();
          out.close();
          socket.close();
          return;
        }

        /**
         * If it is a connect message, return the election info.  Create a
         * Vector of Elections and send it to the client, who has a
         * setElections() method to display the info, which takes in a Vector
         * of elections.
         * When doing this, reset all vote counts to 0, and set all elections
         * as closed, so the
         */
        else if (msg.getType() == Message.CONNECT) {
          //Send info on all elections
          Vector elections = electionsInfo.getElectionCopies(true);
          out.writeObject(Message.getElectionsInfoMessage(elections));
        }
        /**
         * If a vote message is received, check if the candidate and election
         * exist, the election is open, and the user has not already voted in
         * the election.  If all is well, record the vote and send a
         * "vote cast" message.  Otherwise, send an error message.
         */
        else if (msg.getType() == Message.VOTE) {
          //process vote, send confirm/error msg
          String response = "";
          Election e = electionsInfo.getElection(msg.getElectionName());
          if (e == null)
            response = "Election " + e.electionName + " does not exist!";
          else if (!e.isOpen())
            response = "Election " + e.electionName + " is closed.";
          else if (e.userVoted(msg.getUsername()))
            response = "You have already voted in election " + e.getName();
          else {
            if (e.castVote(msg.getCandidateName(), msg.getUsername()))
              response = "Vote successfully cast.";
            else
              response = "Candidate does not exist!";
          }
          out.writeObject(Message.getTextMessage(response));
        }
      }//END while
    }//END try
    catch(Exception e) {
      System.err.println(e.getMessage());
//      e.printStackTrace();
    }
    //Close all streams and sockets
    finally {
      try { in.close(); }
      catch(Exception e) {}

      try { out.close(); }
      catch(Exception e) {}

      try { socket.close(); }
      catch(Exception e) {}
    }
  }


}