import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.util.*;
import java.io.*;


public class ElectionServer
    extends JFrame
    implements ActionListener, ListSelectionListener {

  //GUI components
  //These JLists display the Election and Candidate objects
  private JList candList, electList;
  //These JScrollPanes hold the candidate and election JLists
  private JScrollPane candScroll, electScroll;
  private JButton open, close, refresh;

  //Width and height fields
  private static int BUTTON_WIDTH = 140;
  private static int SCROLL_WIDTH = 250;
  private static int SCROLL_HEIGHT = 350;
  //The election file name
  private final String electFileName = "ElectFile.txt";

  //A place to store all election info
  private ElectionsInfo electionsInfo = new ElectionsInfo();

  //Constructor
  public ElectionServer() {
    //Add a main JPanel to the frame
    JPanel mainPanel = new JPanel(new GridLayout(1,3));
    this.getContentPane().add(mainPanel);

    //Create the election and candidate lists and scroll panes, and add the
    //scroll panes to the main panel
    //Create election list/scroll pane. Set borders and single-selection model
    electList = new JList();
    electList.addListSelectionListener(this);
    electList.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED));
    electList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    electScroll = new JScrollPane(electList);
    //Create candidate list/scroll pane. Set borders and single-selection model:
    candList = new JList();
    candList.addListSelectionListener(this);
    candList.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED));
    candList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    candScroll = new JScrollPane(candList);
    //Create borders for the scroll panes
    electScroll.setBorder(BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(),
        "Elections"));
    candScroll.setBorder(BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(),
        "Candidates"));
    //Set preferred sizes
    electScroll.setPreferredSize(new Dimension(SCROLL_WIDTH, SCROLL_HEIGHT));
    candScroll.setPreferredSize(new Dimension(SCROLL_WIDTH, SCROLL_HEIGHT));
    //Add the Scroll panes
    mainPanel.add(electScroll);
    mainPanel.add(candScroll);

    //Create the button panel and add to main panel
    JPanel buttonPanel = new JPanel(new GridLayout(3,1));
    mainPanel.add(buttonPanel);

    //Create the buttons and add to the button panel
    open = new JButton("Open Election");
    open.addActionListener(this);
    close = new JButton("Close Election");
    close.addActionListener(this);
    refresh = new JButton("Refresh");
    refresh.addActionListener(this);
    //Add to button panel
    buttonPanel.add(open);
    buttonPanel.add(close);
    buttonPanel.add(refresh);
    //Set them enabled/disabled
    setButtonStates();

    //Pack and show the frame
    setTitle("eLection Server");
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    setResizable(false);
    pack();
    show();

    //Finally, read in the ElectFile.txt and listen for incoming connections
    setElections(getElections());
    ConnectionMonitor monitor = new ConnectionMonitor(this, electionsInfo);
    monitor.start();
  }

  //Returns the selected Election, or null if none is selected
  private Election getSelectedElection() {
    return (Election) electList.getSelectedValue();
  }

  //Returns the selected Candidate, or null if none is selected
  private Candidate getSelectedCandidate() {
    return (Candidate) candList.getSelectedValue();
  }



  /**
   * Implement the actionPerformed() method, defined in the ActionListener
   * interface.  This method will process button pushes
   * @param e The ActionEvent to process
   */
  public void actionPerformed(ActionEvent e) {
    //OPEN button - open the selected election and reset votes
    if(e.getSource() == open) {
      Election selected = getSelectedElection();
      //Make sure an election is chosen and is NOT open
      if(selected != null && !selected.isOpen()) {
        selected.open();
//        selected.resetVotes();
//        selected.resetVoterRecord();
        //Update GUI: repaint election and candidate lists, update button states
        electList.repaint();
        candList.repaint();
        setButtonStates();
      }
    }
    //CLOSE button - close the election
    else if(e.getSource() == close) {
      Election selected = getSelectedElection();
      //Make sure an election is chosen and is open
      if(selected != null && selected.isOpen()) {
        selected.close();
        //Update the GUI: repaint the election list and update button states
        electList.repaint();
        setButtonStates();
      }
    }
    //REFRESH button - refreshes the vote count shown on screen
    else if (e.getSource() == refresh) {
      Election selected = getSelectedElection();
      if(selected != null)
        System.out.println("Refreshing election: " + selected.getName());
      candList.repaint();
    }
  }



  /**
   * Implement the actionPerformed() method, defined in the
   * ListSelectionListener interface.  This method will take care of the
   * selection of different list items
   * @param e The ListSelectionEvent to process
   */
  public void valueChanged(ListSelectionEvent e) {
    //Just call updateCandidateList() to list the candidates for the currently
    //selected election
    if(e.getSource() == electList && electList.getModel().getSize() > 0) {
      setButtonStates();
      updateCandidateList();
    }
  }



  /**
   * Sets buttons to be enabled or disabled based on the state of the
   * selected Election
   */
  private void setButtonStates() {
    Election selected = getSelectedElection();
    if(selected == null) {
      open.setEnabled(false);
      close.setEnabled(false);
    }
    else {
      open.setEnabled(!selected.isOpen());
      close.setEnabled(selected.isOpen());
    }
  }


  /**
   * Clears the hashtable of elections, populates it with a new set of
   * Election objects, and finally updates the candidate list
   * @param elections Vector of Election objects to add to the hashtable
   */
  public synchronized void setElections(Vector elections) {
    electionsInfo.clear();
//    electionHash.clear();

    //Iterate through the Vector of elections, placing each in the hashtable
    Election e;
    for(int i=0; i<elections.size(); i++) {
      //Only add the current object if it's an Election object
      if(elections.get(i) instanceof Election) {
        //Save the Election object
        electionsInfo.addElection((Election) elections.get(i));
      }
    }
    //Update the election and candidate lists
    electList.removeAll();
    electList.setListData(elections);
    //Select the first election.  This will cause valueChanged() to be called,
    //which will then call updateCandidateList
    if(electList.getModel().getSize() > 0)
      electList.setSelectedIndex(0);
  }



  /**
   * Redraws the list of candidates to reflect the candidates in the chosen
   * election
   * @param electionIndex the index in the JList of the chosen election
   */
  private void updateCandidateList() {
    ListModel data = electList.getModel();
    Election selected = getSelectedElection();
    //return if no election was selected
    if(selected == null) return;

    //Remove all candidates, and add the new ones in
    candList.removeAll();
    candList.setListData(new Vector(selected.getCandidateHash().values()));

    //Choose the first candidate in the candidate list
    candList.setSelectedIndex(0);
  }


  /**
   * Reads the ElectFile.txt to create a Vector of Election objects for the
   * elections
   * @return a Vector of Elections
   */
  private Vector getElections() {
    Vector electionVect = new Vector();
    try {
      File file = new File(electFileName);
      BufferedReader reader = new BufferedReader(new FileReader(file));

      //Read in each line, and get the election name and candidates
      //Create a new Election object, with Candidate objects inside it
      String line;
      while((line = reader.readLine()) != null) {
        //Get the election name
        StringTokenizer st = new StringTokenizer(line, ",");
        Election election = new Election(st.nextToken());
        //Get all candidates
        while(st.hasMoreTokens()) {
          election.addCandidate(new Candidate(st.nextToken()));
        }
        //Finally, add the election to the vector!
        electionVect.add(election);
      }
    }
    //Catch and print any errors
    catch(FileNotFoundException fnfe) {
      System.out.println("Can't find file " + electFileName);
    }
    catch(IOException ioe) {
      System.out.println(ioe.getMessage());
    }

    //Return the populated Vector of elections
    return electionVect;
  }



  //Main function
  public static void main(String[] args) {
    ElectionServer client = new ElectionServer();
  }


}


