import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.util.*;
import java.io.*;
import java.net.*;


//Defines the GUI for the voting client program
public class ElectionClient
    extends JFrame
    implements ActionListener, ListSelectionListener {

  //GUI components:
  //These JList hold the names (as String objects) of candidates and elections
  private JList candList, electList;
  //These JScrollPanes hold the candidate and election JLists
  private JScrollPane candScroll, electScroll;
  private JButton connect, disconnect, vote;

  //Connection state
  private boolean connected = false;

  //Width and height fields
  private static int SCROLL_WIDTH = 250;
  private static int SCROLL_HEIGHT = 350;
  //The election file name
  private final String electFileName = "ElectFile.txt";

  //A hashtable to hold the Election objects
  Hashtable electionHash = new Hashtable();

  //The thread that will communicate with the server
  private ClientThread thread;
  //The host on which the server resides
  private String host = "localhost";
  //The port the server is listening on
  private int port = 7777;

  //Constructor
  public ElectionClient() {
    //Add a main JPanel to the frame
    JPanel mainPanel = new JPanel(new GridLayout(1,3));
    this.getContentPane().add(mainPanel);

    //Create the election and candidate lists and scroll panes, and add the
    //scroll panes to the main panel
    //Create election list/scroll pane. Set borders and single-selection model
    electList = new JList();
    electList.addListSelectionListener(this);
    electList.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED));
    electList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    electScroll = new JScrollPane(electList);
    //Create candidate list/scroll pane. Set borders and single-selection model:
    candList = new JList();
    candList.addListSelectionListener(this);
    candList.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED));
    candList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    candScroll = new JScrollPane(candList);
    //Create borders for the scroll panes
    electScroll.setBorder(BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(),
        "Elections"));
    candScroll.setBorder(BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(),
        "Candidates"));
    //Set preferred sizes
    electScroll.setPreferredSize(new Dimension(SCROLL_WIDTH, SCROLL_HEIGHT));
    candScroll.setPreferredSize(new Dimension(SCROLL_WIDTH, SCROLL_HEIGHT));
    //Add the Scroll panes
    mainPanel.add(electScroll);
    mainPanel.add(candScroll);

    //Create the button panel and add to main panel
    JPanel buttonPanel = new JPanel(new GridLayout(3,1));
    mainPanel.add(buttonPanel);

    //Create the buttons and add to the button panel
    connect = new JButton("Connect");
    connect.addActionListener(this);
    disconnect = new JButton("Disconnect");
    disconnect.addActionListener(this);
    vote = new JButton("Vote");
    vote.addActionListener(this);
    //Add to button panel
    buttonPanel.add(connect);
    buttonPanel.add(disconnect);
    buttonPanel.add(vote);
    //Set them enabled/disabled
    setButtonStates();

    //Pack and show the frame
    setTitle("eLection Client");
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    setResizable(false);
    pack();
    show();
  }



  /**
   * Implement the actionPerformed() method, defined in the ActionListener
   * interface.  This method will process button pushes
   * @param e The ActionEvent to process
   */
  public void actionPerformed(ActionEvent e) {
    //CONNECT button - pop up a connection dialog box.  Connect only if the
    //user presses the OK button
    if(e.getSource() == connect) {
      //Get username and password
      String title = "Voter Authentication";
      String authMsg = "Enter username and password to connect";
      AuthenticationDialog connDialog = new AuthenticationDialog(title, authMsg);
      int returnVal = connDialog.showAddVoterDialog();
      //Connect only if the user pressed OK.
      if(returnVal == AuthenticationDialog.OK_OPTION) {
        try {
          //Start the thread to communicate with the server
          Socket s = new Socket(host, port);
          thread = new ClientThread(s, this);
          thread.start();
          Message msg = Message.getConnectMessage(connDialog.getUserName(),
                                                  connDialog.getPassword());
          thread.sendMessage(msg);
          setConnectionStatus(true);
        }
        //Display any errors
        catch(Exception ex) {
          String msg;
          if(ex instanceof UnknownHostException)
            msg = "Can't find host: " + host;
          else
            msg = "Could not connect to server!";

          JOptionPane.showMessageDialog(this, msg, "Connection Unsuccessful",
                                        JOptionPane.ERROR_MESSAGE);
          System.err.println(ex.getMessage());
        }
      }
    }

    //DISCONNECT button - closes connection, deletes contents of the lists
    else if(e.getSource() == disconnect) {
      //First, disconnect from server
      thread.disconnect();
      //Take care of gui
      setConnectionStatus(false);
      System.out.println("Disconnected from server");
    }

    //VOTE button - pop up a message box saying that a vote was cast, in
    //which election, and for which candidate
    else if (e.getSource() == vote) {
      //Notify user if either a candidate or election is not chosen
      if (electList.getSelectedIndex() == -1 ||
          candList.getSelectedIndex() == -1) {
        String msg = "Make sure both an election\nand candidate are chosen!";
        //Finally, show the message to the user
        JOptionPane.showMessageDialog(this, msg, "Vote information",
                                      JOptionPane.INFORMATION_MESSAGE);
      }
      //Election and candidate chosen. Send a vote message
      else {
        //Get username and password
        String title = "Voter Authentication";
        String authMsg = "Username and password required to vote";
        AuthenticationDialog connDialog = new AuthenticationDialog(title, authMsg);
        int returnVal = connDialog.showAddVoterDialog();
        //Vote only if the user pressed OK.
        if (returnVal == AuthenticationDialog.OK_OPTION) {
          String elect = ((String) electList.getSelectedValue()).trim();
          String cand = ( (String) candList.getSelectedValue()).trim();
          thread.sendMessage(Message.getVoteMessage(connDialog.getUserName(),
              connDialog.getPassword(), elect, cand));
        }
      }
    }

  } //END actionPerformed()



  /**
   * Implement the actionPerformed() method, defined in the
   * ListSelectionListener interface.  This method will take care of the
   * selection of different list items
   * @param e The ListSelectionEvent to process
   */
  public void valueChanged(ListSelectionEvent e) {
    //Just call updateCandidateList() to list the candidates for the currently
    //selected election
    if(e.getSource() == electList && electList.getModel().getSize() > 0) {
      updateCandidateList();
    }
  }



  /**
   * Sets buttons to be enabled or disabled based on the state of the
   * connection to the server
   */
  private void setButtonStates() {
    connect.setEnabled(!connected);
    disconnect.setEnabled(connected);
    vote.setEnabled(connected);
  }


  /**
   * Saves the state of the connection to the server
   * @param connected whether or not the client is connected to the server
   */
  protected synchronized void setConnectionStatus(boolean connected) {
    this.connected = connected;
    setButtonStates();
    //Empty election and candidate lists if disconnected
    if(!connected) {
      electList.setListData(new Vector());
      candList.setListData(new Vector());
    }
  }


  /**
   * Clears the hashtable of elections, populates it with a new set of
   * election names (Strings), and finaly updates the list of candidate names
   * @param elections Vector of Election objects to add to the hashtable
   */
  public void setElections(Vector elections) {
    electionHash.clear();

    //Since we aren't showing the election status, we need to only display the
    //election names.  Thus, we need a Vector of Strings
    Vector names = new Vector();
    //Iterate through the Vector of electoins, placing each in the hashtable
    Election e;
    for(int i=0; i<elections.size(); i++) {
      //Only add the current object if it's an Election object
      if(elections.get(i) instanceof Election) {
        e = (Election) elections.get(i);
        //Place the Election in the hashtable.  The key is the election name
        electionHash.put(e.getName(), e);
        //Record the election names
        names.add(e.getName());
      }
    }
    //Draw them in the lists
    electList.removeAll();
    electList.setListData(names);
    //Select the first election in the list.  This will trigger
    //valueChanged() to be called, which will call updateCandidateList(),
    //thus updating the candidate list itself
    if(electList.getModel().getSize() > 0)
      electList.setSelectedIndex(0);
  }



  /**
   * Redraws the list of candidates to reflect the candidates in the chosen
   * election
   * @param electionIndex the index in the JList of the chosen election
   */
  private void updateCandidateList() {
    //Get the election name
    String electName = (String) electList.getSelectedValue();
    //Then get the Election object, based on the name, from the hashtable
    Election election = (Election) electionHash.get(electName);

    //Remove all candidate names, and add the new ones in
    candList.removeAll();
    candList.setListData(election.getCandidateNames());

    //Choose the first candidate in the candidate list
    candList.setSelectedIndex(0);
  }



  /**
   * Reads the ElectFile.txt to create a Vector of Election objects for the
   * electi
   * ons
   * @return a Vector of Elections
   */
  public Vector getElections() {
    Vector electionVect = new Vector();
    try {
      File file = new File(electFileName);
      BufferedReader reader = new BufferedReader(new FileReader(file));

      //Read in each line, and get the election name and candidates
      //Create a new Election object, with Candidate objects inside it
      String line;
      while((line = reader.readLine()) != null) {
        //Get the election name
        StringTokenizer st = new StringTokenizer(line, ",");
        Election election = new Election(st.nextToken());
        //Get all candidates
        while(st.hasMoreTokens()) {
          election.addCandidate(new Candidate(st.nextToken()));
        }
        //Finally, add the election to the vector!
        electionVect.add(election);
      }
    }
    //Catch and print any errors
    catch(FileNotFoundException fnfe) {
      System.err.println("Can't find file " + electFileName);
    }
    catch(IOException ioe) {
      System.err.println(ioe.getMessage());
    }

    //Return the populated Vector of elections
    return electionVect;
  }


  //Main function just creates an instance of the gui
  public static void main(String[] args) {
    ElectionClient client = new ElectionClient();
  }


}