

import java.io.*;
import java.util.*;

/**
 * Class that represents an election.  Contains a name, a hashtable of
 * candidates, and a boolean indicating if the election is open or not.
 */
public class Election implements Serializable {
  //Class fields
  protected String electionName;
  protected Hashtable candidateHash;
  protected Hashtable voterRecord;
  protected boolean open;

  //Constructor.  Takes in a name and a hashtable of candidates.
  //Initially, all elections are closed
  public Election(String name) {
    electionName = name;
    candidateHash = new Hashtable();
    voterRecord = new Hashtable();
    open = false;
  }


  //Get the name of the election
  public synchronized String getName() {
    return new String(electionName);
  }


  //Getter and setter function
  protected Hashtable getCandidateHash() {
    return candidateHash;
  }


  //Gets a Vector of candidate names as Strings
  public synchronized Vector getCandidateNames() {
    Vector names = new Vector();
    Enumeration enum = candidateHash.keys();
    //Iterate through candidates, add each name to vector
    while(enum.hasMoreElements()) {
      names.add((String) enum.nextElement());
    }
    return names;
  }


  //Getter and setter operations for open status
  private synchronized void setOpen(boolean open) { this.open = open; }
  public synchronized void open()   {
    open = true;
    resetVotes();
    resetVoterRecord();
  }
  public synchronized void close() { open = false; }
  public synchronized boolean isOpen() { return open; }


  /**
   * Checks if the voter with the given username has voted in this election
   * @param username the username of the voter
   * @return true if the vote has voted in this election, false otherwise
   */
  public synchronized boolean userVoted(String username) {
    return (voterRecord.get(username) != null);
  }


  /**
   * Casts a vote for the given candidate.  If the candidate exists, the
   * election is open and the user has not voted in it, the vote will be
   * successfully cast and the method will return true.
   * Returns false if the election is closed, candidate does not exist, or
   * the user has already voted in this election
   * @param candidate the candidate's name
   * @param username the voter's username
   * @return true if the vote can be successfully cast, false otherwise
   */
  public synchronized boolean castVote(String candidate, String username) {
    //Return false if the election is closed
    if(!this.open) return false;
//    //Remove any whitespace, which might affect the hash function of the hashtable
//    candidate.trim();
//    username.trim();
    //Check if the user already voted in this election
    if(userVoted(username)) return false;
    //Get Candidate
    Object obj = candidateHash.get(candidate);
    if(!(obj instanceof Candidate)) return false;
    Candidate cand = (Candidate)obj;
    //Return false if the candidate does not exist
    if(cand == null) return false;
    //Cast the vote and return true
    cand.incVote();
    voterRecord.put(username, username);
    return true;
  }



  //This function resets all votes for all candidates to 0.  This is
  //useful when an election has been reopened
  public synchronized void resetVotes() {
    Enumeration enum = candidateHash.keys();
    while (enum.hasMoreElements()) {
      String candName = (String)enum.nextElement();
      ((Candidate)candidateHash.get(candName)).setVoteCount(0);
    }
  }


  //Clears the list of usernames of voters who have voted in this election
  public synchronized void resetVoterRecord() {
    voterRecord.clear();
  }


  /**
   * Adds a candidate to the election if the candidate does not exist in
   * in the election.
   * @param cand the Candidate object to add
   * @return a boolean indicating if the candidate was added
   */
  public synchronized boolean addCandidate(Candidate cand) {
    //If the candidate exists, just return false
    String name = cand.getName();
    if(candidateHash.contains(name))
      return false;
    //Otherwise, add the candidate and return true
    candidateHash.put(cand.getName(), cand);
    return true;
  }



  /**
   * Returns a vector of clones of all Candidate objects in this Election
   * @return the Vector of Candidate objects
   */
  public synchronized Vector getCandidateCopies() {
    Vector v = new Vector();
    Iterator i = candidateHash.values().iterator();
    //Add candidate copies to the vector
    while(i.hasNext()) {
      Object obj = i.next();
      if(obj instanceof Candidate)
        v.add( new Candidate((Candidate)obj) );
    }
    return v;
  }

  /**
   * Returns an Election object that is a deep copy of the current Election
   * object, meaning that it preserves its set of candidates.  If hideInfo
   * is true, it will set the election to closed and the vote counts to 0.
   * This is useful any time we want to keep votes secret, such as when
   * sending this object to a client
   * @param hideInfo boolean indicating if we should hide candidate vote counts
   * @return an Election object that is a deep copy of the current object
   */
  public synchronized Election copy(boolean hideInfo) {
    //Get a new Election object
    Election copy = new Election(this.getName());
    copy.setOpen(this.open);
    //Copy all Candidates
    Iterator cands = this.getCandidateCopies().iterator();
    while(cands.hasNext()) {
      Object obj = cands.next();
      if(obj instanceof Candidate) {
        copy.addCandidate((Candidate) obj);
      }
    }
    //If hiding info, reset vote counts and voter records
    if(hideInfo) {
      copy.resetVotes();
      copy.resetVoterRecord();
    }
    //Copy the voterRecord if not hiding info
    else {
      Enumeration enum = this.voterRecord.keys();
      while(enum.hasMoreElements()) {
        Object username = enum.nextElement();
        if(username instanceof String)
          copy.voterRecord.put(username, username);
      }
    }
    return copy;
  }


  /**
   * Returns a String representation of this Election object.  For use in
   * with the JList of elections
   */
  public synchronized String toString() {
    String s = electionName + "  ";
    if (open) return s += "(OPEN)";
    else      return s += "(CLOSED)";
  }


  /**
   * Prints the election name and candidates, separated by commas.  The election
   * name appears first
   */
  public synchronized void print() {
    System.out.print(electionName);
    Iterator i = candidateHash.values().iterator();
    while(i.hasNext()) {
      System.out.print(", " + ((Candidate)i.next()).getName());
    }
    System.out.println();
  }

}

