import java.net.*;
import java.io.*;
import javax.swing.JOptionPane;

public class ClientThread extends Thread {
  private Socket socket;
  private MessageQueue messages;
  private ElectionClient gui;

  //Constructor takes in the Socket to communicate through
  public ClientThread(Socket s, ElectionClient clientGUI) {
    socket  = s;
    gui = clientGUI;
    messages = new MessageQueue();
  }

  /**
   * Overrise the run() method from Thread so this thread can handle
   * communication with the server
   */
  public void run() {
    if(socket == null) return;
    ObjectInputStream in = null;
    ObjectOutputStream out = null;

    try {
      //Get object streams for reading/writing
      out = new ObjectOutputStream(socket.getOutputStream());
      in = new ObjectInputStream(socket.getInputStream());

      while (true) {
        try {
          if (messages.isEmpty()) {
            //Have the thread wait on itself.  This way the sendMessage() method
            //can notify the thread to wake up
            synchronized(this) {
              wait();
            }
          }
        }
        catch (InterruptedException ie) {}
        //Loop back and wait() if there are no messages to send
        if(messages.isEmpty()) continue;

        //Write out the message and read in the response
        Message msg = messages.getNextMessage();
        out.writeObject(msg);
//        System.out.println("sending: " + msg.toString());

        //Exit if the message is a disconnect message.  Streams and socket will
        //be closed in the 'finally' clause at the end of the method
        if(msg.getType() == Message.DISCONNECT) {
          return;
        }

        //Read in the response from the server
        Message response = (Message)in.readObject();
//        System.out.println("got back: " + response.toString());

        //If we are trying to connect, we'll expect and ELECTIONS_INFO message
        //coming back from the server
        if(msg.getType() == Message.CONNECT) {
          if(response.getType() == Message.ELECTIONS_INFO)
            gui.setElections(response.elections);
          else
            System.err.println("Expecting elections info response, got " +
                               response.toString());

        }
        //If we just voted, we should get back a simple text message to display
        else if(msg.getType() == Message.VOTE) {
          if(response.getType() == Message.TEXT_MSG) {
            JOptionPane.showMessageDialog(gui,
                                          response.getTextMessage(),
                                          "Vote - Response from Server",
                                          JOptionPane.INFORMATION_MESSAGE);
          }
          else System.err.println("Expecting text message response, got " +
                                  response.toString());
        }
      }//END while-loop
    }
    catch(Exception e) {
      e.printStackTrace();
    }
    //Set connection status to disconnected, and close all streams and sockets
    finally {
      //Close all streams and sockets
      try { in.close(); }
      catch(Exception e) {}

      try { out.close(); }
      catch(Exception e) {}

      try { socket.close(); }
      catch(Exception e) {}
    }
  }


  //Adds a message to the queue of messages to send
  public synchronized void sendMessage(Message m) {
    messages.addMessage(m);
    //Notify the thread (which is waiting on itself), so that it wakes up
    notifyAll();
  }

  //To disconnect, put a disconnect message in the very front of the queue
  public synchronized void disconnect() {
    messages.prependDisconnectMessage();
    //Notify the thread (which is waiting on itself), so that it wakes up
    notifyAll();
  }
}

