import static spark.Spark.get;
import static spark.Spark.port;
import static spark.Spark.post;

import java.util.Set;

import com.google.gson.Gson;

/**
 * Represents: a server which responds to HTTP requests.
 */
public class Server {

	/**
	 * The message returned by GET /message
	 */
	private String message = "You'll never figure out how to change me";

	/**
	 * Effect: starts running a server
	 */
	public void run() {
		// Make the server run on port 8080
		port(8080);

		// Create a new GSON converter
		Gson gson = new Gson();

		/**
		 * This instructs spark to respond to a GET request. This is a lambda
		 * expression. It is syntactic sugar for:
		 * 
		 * <code>
		 * 	get("/", new Route() {
		 * 		public Object handle(Request request, Response response) throws Exception {
		 * 			return "Hello World";
		 * 		} 
		 * 	});
		 * </code>
		 *
		 * Several defaults are used here. The Content-Type is set to text/html and the
		 * response status is set to 200.
		 */
		get("/", (request, response) -> "Hello World");

		get("/message", (request, response) -> message);

		/**
		 * This is how to send a JSON-encoded object. gson.toJson takes an object and
		 * turns it into a JSON string. It is syntactic sugar for
		 *
		 * <code>
		 * 	get("/json", (request, response) -> {
		 * 		response.type("application/json");
		 * 		return gson.toJson(new MessageBundle("I am a JSON!", 42));
		 * 	});
		 * </code>
		 *
		 * Spark will call the response transformer on the object returned by the Route.
		 * With lambda expressions, however, you can simply write gson::toJson which is
		 * a method reference. gson::toJson has the same signature as the only method in
		 * the interface ResponseTransformer. You can substitute it for a class
		 * implementing the interface.
		 */
		get("/json", (request, response) -> {
			response.type("application/json");
			return new MessageBundle("I am a JSON!", 42);
		}, gson::toJson);

		// Example of how to get query parameters from the request
		// Try sending it a request that has a param named "foo"
		get("/params", (request, response) -> {
			response.type("application/json");
			return new QueryInfo(request.queryString(), request.queryParams(), request.queryParams("foo"));
		}, gson::toJson);

		// Example of getting the body of a post request
		post("/post", (request, response) -> request.body());

		// Wonder what this does...
		post("/message", (request, response) -> {

			response.type("application/json");

			String json = request.body();
			MessageBundle newMessage = gson.fromJson(json, MessageBundle.class);

			if (newMessage != null && newMessage.param == 2112) {
				message = newMessage.message;
				return new MessageSet(true);
			} else {
				response.status(400);
				return new MessageSet(false);
			}

		}, gson::toJson);
	}

	/*
	 * These classes are used for converting to and from JSON. For example, the
	 * object new MessageSet(true) would be converted to {"status":"true"} in JSON.
	 * The field name is taken from the class using reflection.
	 */

	@SuppressWarnings("unused")
	private class MessageSet {
		private boolean status;

		public MessageSet(boolean status) {
			this.status = status;
		}
	}

	@SuppressWarnings("unused")
	private class QueryInfo {
		private String query;
		private Set<String> paramNames;
		private String fooParam;

		public QueryInfo(String query, Set<String> paramNames, String fooParam) {
			this.query = query;
			this.paramNames = paramNames;
			this.fooParam = fooParam;
		}
	}

	class MessageBundle {
		String message;
		int param;

		MessageBundle(String m, int i) {
			message = m;
			param = i;
		}
	}
}
