import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * A singly linked list
 *
 * @param <E> Type to store in each node
 */
public class LinkedList<E> implements Iterable<E> {

	/**
	 * Head and tail of this linked list, or null if this list is empty
	 */
	private Node head, tail;

	/**
	 * Number of elements in this list
	 */
	private int size;

	/**
	 * Current modification count
	 */
	private int modCount;

	/**
	 * Construct an empty linked list
	 */
	public LinkedList() {
		head = tail = null;
		size = 0;
		modCount = 0;
	}

	/**
	 * Add an element to the end of this list
	 * 
	 * @param e Element to add
	 * @return True iff the element was added successfully
	 */
	public boolean add(E e) {
		Node node = new Node(e);
		if (head == null) {
			head = tail = node;
		} else {
			tail.next = node;
			tail = node;
		}
		size++;
		modCount++;
		return true;
	}

	public boolean contains(Object o) {
		for (Node node = head; node != null; node = node.next) {
			if (node.value.equals(o)) {
				return true;
			}
		}
		return false;
	}

	public boolean remove(Object o) {
		Node prev = null;
		for (Node node = head; node != null; prev = node, node = node.next) {
			if (node.value.equals(o)) {
				if (prev == null) {
					head = node.next;
				} else {
					prev.next = node.next;
				}
				node = prev;
				size--;
				modCount++;
				return true;
			}
		}
		return false;
	}

	public int size() {
		return size;
	}

	@Override
	public Iterator<E> iterator() {
		return new LinkedListIterator();
	}

	/**
	 * Node in singly linked list
	 */
	private class Node {

		/** Value stored at this node */
		E value;
		/** Pointer to next node, or null if end of list */
		Node next;

		/**
		 * Construct a new node
		 * 
		 * @param value Value to store at this node
		 */
		private Node(E value) {
			this.value = value;
		}
	}

	/**
	 * Iterator over this linked list
	 */
	private class LinkedListIterator implements Iterator<E> {

		/**
		 * Next node we have yet to return
		 */
		Node next;

		/**
		 * Modification count when this iterator was created
		 */
		int modCount;

		/**
		 * Construct a new linked list iterator
		 */
		private LinkedListIterator() {
			next = head;
			modCount = LinkedList.this.modCount;
		}

		@Override
		public boolean hasNext() {
			return next != null;
		}

		@Override
		public E next() {
			if (LinkedList.this.modCount != modCount) {
				throw new ConcurrentModificationException();
			}
			if (!hasNext()) {
				throw new NoSuchElementException();
			}
			E returnValue = next.value;
			next = next.next;
			return returnValue;
		}
	}
}
