
import java.util.Date;
import java.util.Random;

import matrix.Matrix;
import matrix.MonitorMatrix;
import matrix.RWMatrix;
import matrix.UnsafeMatrix;

public class MatrixTest {
	static final int SIZE = 500;
	private static final int NUM_READERS = 500;
	private static final int NUM_WRITERS = 100;
	private static final int SUM_REPS = 100;
	int granularity = 1;
	private int readers_done = 0;
	private int writers_done = 0;
	Thread[] readers = new Thread[NUM_READERS];
	Thread[] writers = new Thread[NUM_WRITERS];

	enum MatrixImpl {
		UNSAFE, MONITOR, RWLOCK
	}

	MatrixImpl impl;

	public static void main(String[] args) {
		MatrixImpl impl = MatrixImpl.RWLOCK;
		if (args.length >= 1) {
			switch (args[0]) {
			case "unsafe":
				impl = MatrixImpl.UNSAFE;
				break;
			case "monitor":
				impl = MatrixImpl.MONITOR;
				break;
			case "rwlock":
				impl = MatrixImpl.RWLOCK;
				break;
			default:
				System.err.println("Unknown implementation specified: " + args[0]);
				System.err.println("Defaulting to rwlock");
				break;
			}
		} else {
			System.err.println("Usage: lab11.MatrixTest [ unsafe | monitor | rwlock ]");
			System.err.println("No implementation specified; defaulting to rwlock");
		}
		new MatrixTest(impl).run();
	}

	int num_finished;

	Matrix a;
	Random rnd;

	MatrixTest(MatrixImpl impl) {
		this.impl = impl;
		switch (impl) {
		case UNSAFE:
			a = new UnsafeMatrix(SIZE, SIZE);
			break;
		case MONITOR:
			a = new MonitorMatrix(SIZE, SIZE);
			break;
		case RWLOCK:
			a = new RWMatrix(SIZE, SIZE);
			break;
		}
		rnd = new Random();
		a.randomize(rnd);
	}

	class ReaderJob implements Runnable {
		public void run() {
			for (int i = 0; i < SUM_REPS; i++) {
				a.sum();
			}
			synchronized (MatrixTest.this) {
				num_finished++;
				readers_done++;
				MatrixTest.this.notifyAll();
			}
		}
	}

	class WriterJob implements Runnable {
		public void run() {
			a.swap(rnd);
			synchronized (MatrixTest.this) {
				num_finished++;
				writers_done++;
				MatrixTest.this.notifyAll();
			}
		}
	}

	void run() {

		Date t1 = new Date();
		while (granularity * 100 < NUM_READERS)
			granularity *= 10;

		for (int i = 0; i < NUM_READERS; i++) {
			readers[i] = new Thread(new ReaderJob());
			readers[i].start();
		}
		for (int i = 0; i < NUM_WRITERS; i++) {
			writers[i] = new Thread(new WriterJob());
			writers[i].start();
		}

		synchronized (this) {
			while (num_finished < NUM_READERS + NUM_WRITERS)
				try {
					if (num_finished % granularity == 0) {
						System.out.println("Readers finished: " + readers_done);
						System.out.println("Writers finished: " + writers_done);
					}
					wait();
				} catch (InterruptedException e) {
				}
		}
		System.out.println("All threads have finished.");
		Date t2 = new Date();
		System.out.println("Total time: " + (t2.getTime() - t1.getTime()) + "ms (" + impl + ")");
	}
}
// vim: ts=4
