import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;

/**
 * A binary search tree
 *
 * @param <E> The type of element to store in this tree (must be comparable)
 */
public class BST<E extends Comparable<E>> implements Iterable<E> {

	/**
	 * The root of the binary search tree, or empty if the tree is empty. Tree must
	 * satisfy BST invariant.
	 */
	private Optional<Node> root;

	/**
	 * Number of items in the tree
	 */
	private int size;

	/**
	 * Current modification count
	 */
	private int modCount;

	/**
	 * Construct a new, empty binary search tree
	 */
	public BST() {
		root = Optional.empty();
	}

	/**
	 * Add element e to this binary search tree if it doesn't exist.
	 * 
	 * @param e Element to add to the tree
	 * @return True iff the element was added successfully; false if it already
	 *         existed
	 */
	public boolean add(E e) {
		Node prev = null;
		boolean isLastBranchLeft = false;
		Optional<Node> n = root;
		while (!n.isEmpty()) {
			Node node = n.get();
			int cmpRes = e.compareTo(node.value);
			if (cmpRes < 0) {
				prev = node;
				n = node.left;
				isLastBranchLeft = true;
			} else if (cmpRes > 0) {
				prev = node;
				n = node.right;
				isLastBranchLeft = false;
			} else {
				return false;
			}
		}

		Node node = new Node(e, Optional.ofNullable(prev));
		Optional<Node> newNode = Optional.of(node);
		if (prev == null) {
			root = newNode;
		} else {
			if (isLastBranchLeft) {
				prev.left = newNode;
			} else {
				prev.right = newNode;
			}
		}
		size++;
		modCount++;
		return true;
	}

	@SuppressWarnings("unchecked")
	/**
	 * Check if an element is in the tree
	 * 
	 * @param o The element to check for
	 * @return True iff the tree contains the element, as defined using compareTo
	 */
	public boolean contains(Object o) {
		if (!(o instanceof Comparable<?>))
			return false;
		Comparable<E> co;
		try {
			co = (Comparable<E>) o;
		} catch (ClassCastException e) {
			return false;
		}

		Optional<Node> n = root;
		while (!n.isEmpty()) {
			Node node = n.get();

			int cmpRes = co.compareTo(node.value);
			if (cmpRes < 0) {
				n = node.left;
			} else if (cmpRes > 0) {
				n = node.right;
			} else {
				return true;
			}
		}
		return false;
	}

	@Override
	public Iterator<E> iterator() {
		return new TreeIterator();
	}

	/**
	 * The size of this tree
	 * 
	 * @return The number of elements stored in this tree
	 */
	public int size() {
		return size;
	}

	/**
	 * Represents a single node in the binary search tree, with a backpointer to its
	 * parent.
	 */
	private class Node {
		/**
		 * The value stored at this node
		 */
		E value;

		/**
		 * Pointers to the left and right children of this node, satisfying the BST
		 * invariant, or empty if that child does not exist.
		 */
		Optional<Node> left, right;

		/**
		 * Backpointer to the parent of this node, or empty if this is the root of the
		 * tree.
		 */
		Optional<Node> parent;

		/**
		 * Construct a new binary search tree node
		 * 
		 * @param value  The value to store at this node
		 * @param parent Parent node, or empty if this is the root of a new tree
		 */
		private Node(E value, Optional<Node> parent) {
			this.value = value;
			this.parent = parent;
			left = Optional.empty();
			right = Optional.empty();
		}
	}

	/**
	 * Possible states an iterator conducting an in-order traversal can be in.
	 */
	private enum TreeIteratorState {
		LEFT_UNVISITED, SELF_UNVISITED, RIGHT_UNVISITED, ALL_VISITED
	}

	/**
	 * Iterator that performs an in-order traversal on this BST.
	 * 
	 * Will check for concurrent modification and throw an appropriate exception if
	 * detected.
	 */
	private class TreeIterator implements Iterator<E> {

		/**
		 * <p>
		 * Current state of this iterator object.
		 * </p>
		 * 
		 * <p>
		 * If {@code LEFT_UNVISITED}, then none of the nodes in tree rooted at
		 * {@code this.node} have been traversed yet.
		 * </p>
		 * 
		 * <p>
		 * If {@code SELF_UNVISITED}, then all nodes in the left subtree of
		 * {@code this.node} have been traversed, but {@code this.node} has not been
		 * traversed yet, and neither have any of the nodes in its right subtree.
		 * </p>
		 * 
		 * <p>
		 * If {@code RIGHT_UNVISITED}, then all nodes in the left subtree of
		 * {@code this.node}, as well as {@code this.node} itself, have been traversed,
		 * but none of the nodes in the right subtree of {@code this.node} have been
		 * traversed yet.
		 * </p>
		 * 
		 * <p>
		 * If {@code ALL_VISITED}, then all nodes in the tree rooted at
		 * {@code this.node} have been traversed.
		 * </p>
		 */
		TreeIteratorState state;

		/**
		 * Current node as referenced by the invariant on {@code this.state}, or empty
		 * if there are no nodes left to traverse.
		 */
		Optional<Node> node;

		/**
		 * Modification count of tree when this iterator was created
		 */
		private int modCount;

		/**
		 * Construct a new tree iterator
		 */
		private TreeIterator() {
			state = TreeIteratorState.LEFT_UNVISITED;
			node = BST.this.root;
			modCount = BST.this.modCount;
		}

		@Override
		public boolean hasNext() {

			if (node.isEmpty()) {
				return false;
			}

			switch (state) {

			case LEFT_UNVISITED:
				// Move down the left child until we hit the bottom
				while (!node.get().left.isEmpty()) {
					node = node.get().left;
				}
				// Transition to SELF_UNVISITED
				state = TreeIteratorState.SELF_UNVISITED;
				return true;

			case SELF_UNVISITED:
				return true;

			case RIGHT_UNVISITED:
				// Move into right subtree, and then start over with LEFT_UNVISITED
				if (!node.get().right.isEmpty()) {
					node = node.get().right;
					state = TreeIteratorState.LEFT_UNVISITED;
					return hasNext();
				}
				// If no right child, then we're done with this subtree
				state = TreeIteratorState.ALL_VISITED;

				//
				// Intentional Fall-Through
				//
				// You could recursively call `return hasNext()` here for the same effect,
				// but we're making a minor optimization by taking advantage of switch case
				// fallthrough.
				// If you choose to do this, you'll definitely want to document it like we do
				// here, because otherwise fallthrough is an easy source of bugs.

			case ALL_VISITED:
				// Move up the tree
				while (true) {
					Optional<Node> child = node;
					node = node.get().parent;

					// If we hit the root, we're done traversing the tree
					if (node.isEmpty()) {
						return false;
					}

					// If we came up a left pointer, then this node and its right subtree are
					// unvisited
					if (node.get().left.equals(child)) {
						break;
					}

					// Otherwise, we came up a right pointer, so the entire subtree at this node is
					// visited, meaning we need to keep moving up
				}

				// We only break out of the loop if node and its right subtree are unvisited, so
				// transition to SELF_UNVISITED
				state = TreeIteratorState.SELF_UNVISITED;
				return true;

			}

			return false;
		}

		@Override
		public E next() {

			if (BST.this.modCount != modCount) {
				throw new ConcurrentModificationException();
			}

			if (!hasNext()) {
				throw new NoSuchElementException();
			}

			state = TreeIteratorState.RIGHT_UNVISITED;
			return node.get().value;
		}

	}
}
