package model;

/**
 * A fun game of tic-tac-toe.
 */
public class TicTacToe {

	/**
	 * Dimensions of each side of the board; the win detection assumes this number
	 * is odd
	 */
	private final static int SIZE = 3;

	/**
	 * Possible states of the game
	 */
	public static enum GameStatus {
		X_TURN, O_TURN, X_WIN, O_WIN, TIE
	}

	/**
	 * Current status of the game
	 *
	 * Class Invariant: Must be accurate after every single move.
	 */
	private GameStatus status;

	/**
	 * Tic-tac-toe board. Null represents empty, true is x, and false is o.
	 */
	private Boolean[][] board;

	/**
	 * Initialize a brand new game of tic-tac-toe
	 */
	public TicTacToe() {
		status = GameStatus.X_TURN;
		board = new Boolean[SIZE][SIZE];
	}

	/**
	 * Get current game status
	 * 
	 * @return Status of the game
	 */
	public GameStatus currentStatus() {
		return status;
	}

	/**
	 * Attempt to let the current player move in the specified cell.
	 * 
	 * @param x X coordinate; must be in [0, 2]
	 * @param y Y coordinate; must be in [0, 2]
	 * @return Whether the move was successful
	 */
	public boolean move(int x, int y) {

		switch (status) {
		case X_TURN:
		case O_TURN:
			break;
		default:
			return false;
		}

		if (board[x][y] != null) {
			return false;
		}

		board[x][y] = (status == GameStatus.X_TURN);
		checkWinTie();

		switch (status) {
		case X_TURN:
			status = GameStatus.O_TURN;
			break;
		case O_TURN:
			status = GameStatus.X_TURN;
			break;
		default:
			break;
		}

		return true;
	}

	/**
	 * Sets the current player to be the winner of the game.
	 */
	private void setCurrentPlayerWinner() {
		switch (status) {
		case X_TURN:
			status = GameStatus.X_WIN;
			break;
		case O_TURN:
			status = GameStatus.O_WIN;
			break;
		default:
			throw new IllegalStateException();
		}
	}

	/**
	 * Checks for wins or ties and updates the game status accordingly.
	 */
	private void checkWinTie() {
		// Check for win

		// Check along each axis
		for (int i = 0; i < SIZE; i++) {
			if (board[i][i] == null) {
				continue;
			}

			Boolean candidate = board[i][i];
			boolean horizWin = true;
			boolean vertWin = true;
			for (int j = 0; j < SIZE; j++) {
				if (!candidate.equals(board[i][j])) {
					horizWin = false;
				}
				if (!candidate.equals(board[j][i])) {
					vertWin = false;
				}
			}
			if (horizWin || vertWin) {
				setCurrentPlayerWinner();
				return;
			}
		}
		// Check diagonals
		final int MID = SIZE / 2;
		if (board[MID][MID] != null) {
			Boolean candidate = board[MID][MID];
			boolean winDiag = true;
			boolean winOff = true;
			for (int i = 0; i < SIZE; i++) {
				if (!candidate.equals(board[i][i])) {
					winDiag = false;
				}
				if (!candidate.equals(board[i][SIZE - i - 1])) {
					winOff = false;
				}
			}
			if (winDiag || winOff) {
				setCurrentPlayerWinner();
				return;
			}
		}

		// Check for tie
		for (Boolean[] row : board) {
			for (Boolean cell : row) {
				if (cell == null) {
					return;
				}
			}
		}
		status = GameStatus.TIE;
	}

}
