package game1024;

import java.util.List;

import game1024.GameModel.Move;

public class Search {
   static final int[][] dirs = { { 0, 1 }, { 1, 0 }, { 0, -1 }, { -1, 0 } };

   static public class MoveChoice {
      final int dx, dy;
      int heuristic;

      MoveChoice(int dx, int dy, int val) {
         this.dx = dx;
         this.dy = dy;
         heuristic = val;
      }
   }

   public static MoveChoice bestMove(GameModel g, int depth) {
      MoveChoice m = bestMove(g, depth, -1, Integer.MAX_VALUE);
      if (m != null) {
         System.out.println("Empty cells: " + num_empty_cells(g)
                  + ", clustering score " + clustering_score(g));
         System.out.println("Best move: " + m.heuristic);
      }
      return m;
   }

   /**
    * Return the minmax best move to depth 'depth' if its value is between min
    * and max. Return null if there is no minmax move at least as good as min.
    */
   static MoveChoice bestMove(GameModel g, int depth, int min, int max) {

      MoveChoice best = null;

      for (int dir = 0; dir < 4; dir++) {
         int dx = dirs[dir][0], dy = dirs[dir][1];
         GameModel g2 = g.clone();
         List<Move> moves = g2.doMove(dx, dy);
         if (moves.isEmpty()) continue;
         int gh = (depth > 0) ? averagePlacement(g2, depth, min, max)
                  : heuristic_score(g2);
         if (gh >= max) {
            assert gh != Integer.MAX_VALUE;
            return new MoveChoice(dx, dy, max);
         }
         if (gh >= min) {
            best = new MoveChoice(dx, dy, gh);
            min = gh + 1;
         }
      }

      return best;
   }

   static int worstPlacement(GameModel g2, int depth, int min, int max) {
      assert depth > 0;
      boolean any_empty_cell = false;
      int ret = max;

      for (int r = 0; r < g2.size(); r++)
         for (int c = 0; c < g2.size(); c++) {
            GameModel g3 = g2.clone();
            if (g2.tile(r, c) != 0) continue;
            any_empty_cell = true;
            for (int v = 1; v <= 2; v++) {
               g3.setTile(r, c, v);
               MoveChoice d = bestMove(g3, depth - 1, min, max);
               if (d == null || d.heuristic <= min) {
                  // System.out.println("pruning: " + (d == null ? "null" :
                  // d.value) + " <= " + min);
                  return min;
               }
               if (d.heuristic < ret) {
                  ret = d.heuristic;
                  max = ret - 1;
               }
            }
         }
      return any_empty_cell ? ret : min; // no placement available!
   }

   static int averagePlacement(GameModel g2, int depth, int min, int max) {
      assert depth > 0;

      int sum = 0;
      int num_empty = 0;

      for (int r = 0; r < g2.size(); r++)
         for (int c = 0; c < g2.size(); c++) {
            GameModel g3 = g2.clone();
            if (g2.tile(r, c) != 0) continue;
            num_empty++;
            for (int v = 1; v <= 2; v++) {
               g3.setTile(r, c, v);
               MoveChoice d = bestMove(g3, depth - 1, min, max);
               if (d == null) {
                  sum += min;
               } else {
                  sum += d.heuristic;
               }
            }
         }
      if (num_empty == 0) return min;
      else return sum / 2 / num_empty;
   }

   static int heuristic_score(GameModel g) {
      int rs = g.getScore();
      int ec = num_empty_cells(g);
      int cs = clustering_score(g);
      int h = (int)(rs + (Math.log(rs) + 2) * ec - cs);
      return Math.max(h, Math.min(rs, 1));
   }

   private static int clustering_score(GameModel g) {
      int ret = 0;
      // clustering score is the sum of the individual clustering
      // scores for the occupied tiles
      for (int i = 0; i < g.size(); ++i) {
         for (int j = 0; j < g.size(); ++j) {
            if (g.tile(i, j) != 0) ret += cell_clustering_score(g, i, j);
         }
      }
      return ret;
   }

   private static int cell_clustering_score(GameModel g, int i, int j) {
      int v = g.tile(i, j);
      int n = g.size();
      assert v != 0;
      int sum1 = 0; // things other than immediately smaller and bigger
      int sum2 = 0; // immediately smaller and bigger
      boolean saw_smaller = false;
      boolean saw_bigger = false;
      for (int d = 0; d < 4; d++) {
         int x = i + dirs[d][0];
         int y = j + dirs[d][1];
         if (x < 0 || x >= n || y <= 0 || y >= n) continue;
         int v2 = g.tile(x, y);
         if (v2 == 0) {
            sum2 += v / 8;
            continue;
         }

         if (v2 + v2 == v && !saw_smaller) saw_smaller = true;
         else if (v + v == v2 && !saw_bigger) {
            saw_bigger = true;
         } else {
            if (v2 + v2 == v) sum2 += v2;
            else if (v + v == v2) sum2 += v;
            else sum1 += Math.abs(v - v2);
         }
      }
      if (saw_smaller) return sum2;
      return sum1 + sum2;

      // int neighbors = 0;
      // int sum = 0;
      // int n = g.size();
      // for (int x = i-1; x <= i+1; x++) {
      // if (0 <= x && x < n) {
      // for (int y = j-1; y <= j+1; y++) {
      // if (0 <= y && y < n && g.tile(x,y) != 0) {
      // neighbors++;
      // sum += Math.abs(g.tile(i,j) - g.tile(x,y));
      // }
      // }
      // }
      // }
      // return sum/neighbors;
   }

   private static int num_empty_cells(GameModel g) {
      int ret = 0;
      for (int i = 0; i < g.size(); i++) {
         for (int j = 0; j < g.size(); j++) {
            if (g.tile(i, j) == 0) ret++;
         }
      }
      return ret;
   }

}