package mvcDemo;

import javafx.animation.FadeTransition;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.scene.Node;
import javafx.scene.canvas.Canvas;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.TextField;
import javafx.scene.input.MouseEvent;
import javafx.util.Duration;

/**
 * This is the controller. When the FXML loader loads the FXML
 * file, an instance of this class is created and initialized
 * by running the initialize() method below on the Application
 * thread. This happens automatically.
 */
public class Controller {
   // global constants
   static final int N = 10;
   static final int SIZE = 600;
   
   /**
    * These fields refer to View components. They are populated
    * ('injected') by the FXML loader so that they are available
    * during initialization and handler calls. A component can be
    * injected by giving it an fx:id equal to the field name in
    * the FXML file and decorating the field here with @FXML.
    */
   @FXML private Button startButton;
   @FXML private Button stopButton;
   @FXML private TextField hit;
   @FXML private TextField miss;
   @FXML private Canvas canvas;
   @FXML private Label score;
   @FXML private Label rating;
   
   private Model model;
   private Grid grid;
   private FadeTransition hitTransition;
   private FadeTransition missTransition;
   private Timeline timeline;
   
   /**
    * This method is run automatically after the scene is loaded.
    * Most initialization work for the application can be done here.
    */
   @FXML
   public void initialize() {
      model = new Model();
      grid = new Grid(model, canvas);
      grid.draw();
      startButton.setDisable(false);
      stopButton.setDisable(true);
      hitTransition = createTransition(hit);
      missTransition = createTransition(miss);
      
      // register a listener on the rating label
      model.rating.addListener(new ChangeListener<String>() {
         @Override
         public void changed(ObservableValue<? extends String> observable,
                  String oldValue, String newValue) {
            rating.setText(newValue);            
         }
      });
      
      timeline = new Timeline(new KeyFrame(Duration.millis(100),
               new EventHandler<ActionEvent>() {
                  @Override
                  public void handle(ActionEvent ae) {
                     model.advance();
                     grid.draw();
                  }
               }));
      timeline.setCycleCount(Timeline.INDEFINITE);
      
      /**
       * This is a separate Animation thread that independently
       * periodically changes the state of the animation.
       * The method call in the handler must be synchronized, as it
       * accesses shared data.
       */
      Thread t = new Thread(new Runnable() {
         @Override
         public void run() {
            Timeline tl = new Timeline(new KeyFrame(Duration.millis(4000),
                     new EventHandler<ActionEvent>() {
               public void handle(ActionEvent ae) {
                  model.setRandomDirection();
               }
            }));
            tl.setCycleCount(Timeline.INDEFINITE);
            tl.play();
         }
      });
      t.setDaemon(true);
      t.start();
   }
   
   /**
    * Event handlers. They are associated with the View components
    * in the FXML file.
    */
   
   @FXML
   private void handleButtonPressed(ActionEvent ae) {
      Button button = (Button)ae.getSource();
      boolean run = button.equals(startButton);
      startButton.setDisable(run);
      stopButton.setDisable(!run);
      model.setRunning(run);
      if (run) timeline.play();
      else timeline.pause();
   }

   @FXML
   private void handleMousePressed(MouseEvent e) {
      hitTransition.jumpTo("end");
      missTransition.jumpTo("end");
      int x = (int)(e.getX()*N/SIZE);
      int y = (int)(e.getY()*N/SIZE);
      int currentx = model.getX();
      int currenty = model.getY();
      if (currentx == x && currenty == y) {              
         grid.updateBG();
         grid.draw();
         model.setRandomDirection();
         model.updateScore(true);
         hitTransition.jumpTo("start");
         hitTransition.play();
      } else {
         model.updateScore(false);
         missTransition.jumpTo("start");
         missTransition.play();
      }
      score.setText(model.getScore());
   }

   private FadeTransition createTransition(Node n) {
      FadeTransition ft = new FadeTransition(Duration.millis(1000), n);
      ft.setFromValue(1.0);
      ft.setToValue(0.0);
      return ft;
   }

}
