/*******
 *
 * A JavaFX reimplementation of Gabriele Cirulli's 2048 game
 * 
 */

package game1024;

import game1024.Search.MoveChoice;

import java.util.ArrayList;
import java.util.List;

import javafx.animation.KeyFrame;
import javafx.animation.KeyValue;
import javafx.animation.Timeline;
import javafx.application.Application;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.scene.Scene;
import javafx.scene.input.KeyEvent;
import javafx.scene.layout.Pane;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;
import javafx.scene.text.Text;
import javafx.stage.Stage;
import javafx.util.Duration;

public class Game extends Application {

   static Color numColor = new Color(0.4, 0.4, 0.4, 1.0);
   static Color tileColor = new Color(0.8, 0.8, 0.8, 1.0);
   static Color sepColor = new Color(0.6, 0.6, 0.6, 1.0);
   static Color[] colors = { tileColor, new Color(0.8, 0.8, 0.75, 1.0),
            new Color(0.8, 0.75, 0.75, 1.0), new Color(0.85, 0.75, 0.7, 1.0),
            new Color(0.8, 0.8, 0.7, 1.0), new Color(0.8, 0.7, 0.7, 1.0),
            new Color(0.8, 0.7, 0.6, 1.0), new Color(0.8, 0.6, 0.7, 1.0),
            new Color(0.7, 0.8, 0.7, 1.0), new Color(0.7, 0.8, 0.6, 1.0),
            new Color(0.6, 0.7, 0.8, 1.0), new Color(0.6, 0.6, 1.0, 1.0),
            new Color(0.9, 0.9, 0.5, 1.0), new Color(0.8, 0.7, 0.5, 1.0),
            new Color(0.8, 0.6, 0.5, 1.0), new Color(0.8, 0.5, 0.7, 1.0),
            new Color(0.8, 0.5, 0.5, 1.0) };
   static Font[] numFonts = { Font.font("Helvetica-Bold", FontWeight.BOLD, 64),
            Font.font("Helvetica-Bold", FontWeight.BOLD, 48),
            Font.font("Helvetica-Bold", FontWeight.BOLD, 32),
            Font.font("Helvetica-Bold", FontWeight.BOLD, 24) };
   static double tileSize = 120.0;
   static double gutter;
   static Duration slide_time = Duration.seconds(0.2);
   static Duration flash_time = Duration.seconds(0.1);
   boolean currently_animating = false;
   Timeline current_timeline; // non-null if currently_animating

   // The model of the game.
   private GameModel model;
   // The nodes corresponding to each of the tiles of the game
   private Tile[][] tiles;
   private Text scoreBox;

   public static void main(String[] args) {
      Application.launch(args);
   }

   public void start(Stage stage) {
      model = new GameModel();
      model.addNewTile();

      gutter = tileSize / 10;

      stage.setTitle("1024");
      // Group outer = new Group();

      Pane gamePane = new VBox();
      stage.setScene(new Scene(gamePane));
      Pane tilePane = new Pane();
      gamePane.getChildren().add(scoreBox = new Text("0"));
      gamePane.getChildren().add(tilePane);
      // outer.getChildren().add(gamePane);

      int s = model.size();
      tilePane.setPrefHeight(s * tileSize + (s + 1) * gutter);
      tilePane.setPrefWidth(s * tileSize + (s + 1) * gutter);
      tilePane.setStyle("-fx-background-color: #999999");
      gamePane.setStyle("-fx-background-color: #999999; -fx-padding: 8pt");
      scoreBox.setStyle(
               "-fx-fill: white; -fx-font-size: 24pt; -fx-font-weight: bold");

      tiles = new Tile[model.size()][model.size()];

      for (int i = 0; i < model.size(); i++) {
         for (int j = 0; j < model.size(); j++) {
            Tile tile = new Tile(i, j); // extra blank tile
            tilePane.getChildren().add(tile);
            tile.setPosn();
            tile.toBack();

            tile = tiles[i][j] = new Tile(i, j);
            tilePane.getChildren().add(tile); // j,i
         }
      }
      resetTilePosns();
      setLabelsFromState();
      gamePane.setOnKeyPressed(new ArrowHandler());
      gamePane.requestFocus();
      stage.sizeToScene();
      stage.show();
   }

   /**
    * A tile node on the game board. Keeps track of its contained background and
    * label.
    */
   static class Tile extends StackPane {
      public final Text label;
      public final Rectangle background;
      int row, column;

      public Tile(int row, int column) {
         Rectangle r = background = new Rectangle();
         this.row = row;
         this.column = column;
         label = new Text();
         getChildren().add(r);
         getChildren().add(label);
         r.setWidth(tileSize);
         r.setHeight(tileSize);
         r.setFill(tileColor);
         r.setStrokeWidth(0);
         r.setStroke(tileColor);
         r.setArcWidth(tileSize * 0.30);
         r.setArcHeight(tileSize * 0.30);
         label.setFont(numFonts[0]);
         label.setFill(numColor);
      }

      public void setPosn() {
         setLayoutX(gutter + column * (tileSize + gutter));
         setLayoutY(gutter + row * (tileSize + gutter));
      }
   }

   private void resetTilePosns() {
      for (int i = 0; i < model.size(); i++) {
         for (int j = 0; j < model.size(); j++) {
            tiles[i][j].setPosn();
         }
      }
   }

   private void updateScore() {
      scoreBox.setText(Integer.toString(model.getScore()));
   }

   /** Clean up the UI state after an animation is done. */
   private void finishAnimation() {
      currently_animating = false;
      setLabelsFromState();
      resetTilePosns();
      updateScore();
   }

   /** Animate an attempted move in direction (dx, dy). */
   private void doMove(final int dx, final int dy) {
      if (currently_animating) {
         current_timeline.stop();
         finishAnimation();
      }

      List<GameModel.Move> moves = model.doMove(dx, dy);
      if (moves.isEmpty()) return;

      List<KeyValue> move_anims = new ArrayList<KeyValue>();
      List<KeyValue> flash_anims = new ArrayList<KeyValue>();
      for (GameModel.Move m : moves) {
         Tile from_tile = tiles[m.from.row][m.from.col];
         Tile to_tile = tiles[m.to.row][m.to.col];
         from_tile.toFront();
         move_anims.add(new KeyValue(from_tile.layoutXProperty(),
                  to_tile.getLayoutX()));
         move_anims.add(new KeyValue(from_tile.layoutYProperty(),
                  to_tile.getLayoutY()));
         move_anims.add(new KeyValue(from_tile.background.fillProperty(),
                  from_tile.background.getFill()));
         if (m.merge)
            flash_anims.add(new KeyValue(from_tile.background.fillProperty(),
                     ((Color)(from_tile.background.getFill()))
                              .interpolate(Color.WHITE, 0.3)));
      }

      Timeline timeline = current_timeline = new Timeline(60);
      timeline.getKeyFrames().add(new KeyFrame(slide_time, "end_slide",
               new EventHandler<ActionEvent>() {
                  public void handle(ActionEvent ae) {}
               }, move_anims));

      timeline.getKeyFrames()
               .add(new KeyFrame(
                        flash_anims.isEmpty() ? slide_time
                                 : slide_time.add(flash_time),
                        "end", new EventHandler<ActionEvent>() {
                           @Override
                           public void handle(ActionEvent ae) {
                              finishAnimation();
                           }
                        }, flash_anims));

      currently_animating = true;
      timeline.play();

      return;
   }

   private void setLabelsFromState() {
      for (int r = 0; r < model.size(); r++) {
         for (int c = 0; c < model.size(); c++) {
            int v = model.tile(r, c);
            Text t = tiles[r][c].label;
            if (v == 0) {
               t.setText("");
            } else {
               t.setText(String.valueOf(v));
               t.setFont(v < 10 ? numFonts[0]
                        : v < 100 ? numFonts[1]
                                 : v < 1000 ? numFonts[2] : numFonts[3]);
            }
            int j = 0;
            while (v != 0) {
               v = v / 2;
               j++;
            }
            if (j >= colors.length) j = colors.length - 1;
            tiles[r][c].background.setFill(colors[j]);
         }
      }
   }

   /** Handler for arrow keys to trigger moves. */
   class ArrowHandler implements EventHandler<KeyEvent> {
      @Override
      public void handle(KeyEvent ev) {
         if (ev.getEventType() == KeyEvent.KEY_PRESSED) {
            switch (ev.getCode()) {
            case LEFT:
               doMove(-1, 0);
               break;
            case RIGHT:
               doMove(1, 0);
               break;
            case UP:
               doMove(0, -1);
               break;
            case DOWN:
               doMove(0, 1);
               break;
            case SPACE:
               MoveChoice d = Search.bestMove(model, 2);
               if (d != null) doMove(d.dx, d.dy);
               break;
            default:
               break;
            }
         }
         ev.consume();
      }
   }
}
