package eventgen;

import java.util.Collection;

/** An EventGenerator simulates independent boolean random variables. It keeps
 * track of a set of random variables and their probabilities of being true.
 * When sampled, it returns the set of variables that turned out to be true.
 * Variables can be added to or removed from the set, and their probabilities
 * can be adjusted.
 */
public interface EventGenerator<Variable> {

	/** Add a new variable.
	 * Requires that the variable not already be in the set.
	 */
	void addVariable(Variable e, double probability);
	
	/** Returns whether the variable is in the set. */
	boolean contains(Variable e);
	
	/** Change the probability of a variable that is already in the set. */
	void setProbability(Variable e, double probability);
	
	/** Remove an event that is in the set.
	 *  Requires that e is in the set.
	 */
	void removeVariable(Variable e);

	/** Randomly sample all variables. Returns a collection of the
	 *  variables that sampled as true.
	 */
	Collection<Variable> sample();
	
	/** Randomly sample all variables for a given number of timesteps,
	 *  which may be fractional.  The rate at which variables are true is
	 *  chosen so that over a timestep of 1 they are true with the assigned
	 *  probability: sample(1.0) behaves like sample().  Returns a
	 *  collection of the variables that sampled as true.
	 */
	Collection<Variable> sample(double timestep);
}
