package eventgen;

import java.util.Collection;
import java.util.Date;
import java.util.Random;

/** A test harness for the event generator implementations. Assertion
 * checking should be turned on for the bounded exhaustive checking in
 * test2(). */
public class EventGenTest {
	static class IntArrAlloc implements TreeGenerator.ArrayAllocator<Integer> {
		public Integer[] newArray(int n) {
			return new Integer[n];
		}
	}
	
	static boolean use_treegen = true;
	static boolean random_probs = false;
	
	public static void main(String[] args) {
		test1();
		test2();
	}
	
	static final int N = 100;
	static void test1() {
		System.out.println("Stress/performance test");
		Random r = new Random();
		EventGenerator<Integer> gen;
		if (use_treegen)
			gen = new TreeGenerator<Integer,IntArrAlloc>(new IntArrAlloc(), r);
		else
			gen = new SimpleGenerator<Integer>();
		Integer[] evs = new Integer[N];
		Date start = new Date();
		for (int i = 0; i < N; i++) {
			evs[i] = new Integer(i);
			Collection<Integer> c = gen.sample();
			double p = random_probs ? (1 + (double)r.nextInt(N))/N : 1.0/N;
			gen.addVariable(evs[i], p);
			if (N < 50) {
				System.out.println("" + i + "->" + p);
			}
		}
		int sum = 0, sum2 = 0, reps = 10000;
		for (int i = 0; i < reps; i++) {
			Collection<Integer> c = gen.sample();
			if (reps < 20) { System.out.print(" " + c); System.out.println(); }
			int n = c==null ? 0 : c.size();
			sum += n;
			sum2 += n*n;
		}
		Date end = new Date();
		System.out.println("Total time = " + (end.getTime() - start.getTime()) + "ms");
		System.out.println("Average number of events: " + ((double)sum/reps));
		System.out.println("Variance: " + ((double)sum2/reps - (double)sum*sum/reps/reps));
	}
	
	/* Bounded exhaustive testing of event generator. This works
	 * recursively, trying all possible short sequences of operations
	 * against the event generator. */
	static void test2() {
		System.out.println("Starting bounded exhaustive testing, depth " +
				MAX_TEST_DEPTH);
		Random r = new Random();
		TreeGenerator<Integer,IntArrAlloc> gen =
				new TreeGenerator<Integer,IntArrAlloc>(new IntArrAlloc(), r);
		test2a(gen, 0, r);
		System.out.println("Done.");
	}
	static final int MAX_TEST_DEPTH = 8;
	static void test2a(TreeGenerator<Integer,IntArrAlloc> gen, int depth, Random r) {
		if (depth > MAX_TEST_DEPTH) return;
		Collection<Integer> c = gen.sample();
		TreeGenerator<Integer,IntArrAlloc> gen2 =
				new TreeGenerator<Integer,IntArrAlloc>(gen);
		gen2.addVariable(depth,  (1 + (double)r.nextInt(N))/N);
		test2a(gen2, depth+1, r);
		for (int i = 0; i < depth; i++) {
			TreeGenerator<Integer,IntArrAlloc> gen3 =
					new TreeGenerator<Integer,IntArrAlloc>(gen);
			if (gen3.contains(i))
				gen3.removeVariable(i);
			test2a(gen3, depth+1, r);
		}
		for (int i = 0; i < depth; i++) {
			TreeGenerator<Integer,IntArrAlloc> gen3 =
					new TreeGenerator<Integer,IntArrAlloc>(gen);
			if (gen3.contains(i))
				gen3.setProbability(i, (1 + (double)r.nextInt(N))/N); 
			test2a(gen3, depth+1, r);
		}
	}
}
