package cs2110;

/**
 * Models a checking account in our personal finance app.
 */
public class CheckingAccount extends Account {

    /**
     * The balance must remain above this amount, in cents, to prevent the monthly account fee.
     */
    public static final int MINIMUM_BALANCE = 10000;

    /**
     * The monthly account fee.
     */
    public static final int ACCOUNT_FEE = 500;

    /**
     * Whether the account fee should be charged this month. Will be `false` unless `balance()` ever
     * was below MINIMUM_BALANCE since the previous call to `transactionReport()`
     */
    private boolean chargeFee;

    /**
     * Constructs a checking account with the given `name` and initial `balance`.
     */
    public CheckingAccount(String name, int balance) {
        super(name, balance);
        this.chargeFee = this.balance() < MINIMUM_BALANCE;
    }

    /**
     * Attempts to transfer the specified `amount`, in cents, from this account to
     * `receivingAccount` and returns whether this transaction was successful. If this transaction
     * is successful, it is logged to both accounts with the given `memo`. Otherwise, no changes are
     * made to either account and no transaction is logged. Updates `chargeFee` if `balance` drops
     * below `MINIMUM_BALANCE`. Requires that `amount > 0`.
     */
    @Override
    public boolean transferFunds(Account receivingAccount, int amount) {
        boolean success = super.transferFunds(receivingAccount, amount);
        if (success && this.balance() < MINIMUM_BALANCE) {
            this.chargeFee = true;
        }
        return success;
    }

    /**
     * Debits the monthly account fee if the balance fell below the minimum during the month.
     */
    @Override
    protected void closeOutMonth() {
        if (this.chargeFee) {
            this.withdrawFunds(ACCOUNT_FEE, "Account Maintenance Fee");
        }
        this.chargeFee = this.balance() < MINIMUM_BALANCE; // reset `chargeFee` for the new month
    }
}
