package cs2110;

/**
 * The base class for all account types in our personal finance app.
 */
public abstract class Account {

    /**
     * The name of this account.
     */
    private String name;

    /**
     * The current balance, in cents, of this account.
     */
    private int balance;

    /**
     * The current month's transaction report.
     */
    private StringBuilder transactions;

    /**
     * Constructs an account with the given `name` and initial `balance`.
     */
    public Account(String name, int balance) {
        this.name = name;
        this.balance = balance;
        this.resetTransactionLog();
    }

    /**
     * Reassigns the transaction log to a new `StringBuilder` object that contains this month's
     * initial account balance. This method must be called at the start of every month (i.e.,
     * from the constructor and during the execution of `transactionReport()`).
     */
    private void resetTransactionLog() {
        this.transactions = new StringBuilder("Initial Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n"); // newline
    }

    /**
     * Converts the given number of `cents` into a String in "$X.XX" format.
     */
    protected static String centsToString(int cents) {
        int dollars = cents / 100;
        cents = cents % 100;
        return ("$" + dollars + "." + (cents < 10 ? "0" : "") + cents);
    }

    /**
     * Return the name associated with this account.
     */
    public String name() {
        return this.name;
    }

    /**
     * Returns the total balance, in cents, of the account.
     */
    public int balance() {
        return this.balance;
    }

    /**
     * Attempts to deposit the specified `amount`, in cents, to the balance of the account, and
     * returns whether this transaction was successful. If this transaction is successful (always
     * true in the provided implementation), it is logged with the given `memo`. Otherwise, no
     * changes are made to this account, and no transaction is logged. Requires that `amount > 0`.
     */
    public boolean depositFunds(int amount, String memo) {
        assert amount > 0;
        this.balance += amount;
        this.transactions.append(" - Deposit ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(": ");
        this.transactions.append(memo);
        this.transactions.append("\n");
        return true;
    }

    /**
     * Attempts to withdraw the specified `amount`, in cents, from the balance of the account, and
     * returns whether this transaction was successful. If this transaction is successful (always
     * true in the provided implementation), it is logged with the given `memo`. Otherwise, no
     * changes are made to this account, and no transaction is logged. Requires that `amount > 0`.
     */
    protected boolean withdrawFunds(int amount, String memo) {
        assert amount > 0;
        this.balance -= amount;
        this.transactions.append(" - Withdraw ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(": ");
        this.transactions.append(memo);
        this.transactions.append("\n");
        return true;
    }

    /**
     * Attempts to transfer the specified `amount`, in cents, from this account to
     * `receivingAccount` and returns whether this transaction was successful. If this transaction
     * is successful, it is logged to both accounts with the given `memo`. Otherwise, no changes are
     * made to either account, and no transaction is logged. Requires that `amount > 0`.
     */
    public boolean transferFunds(Account receivingAccount, int amount) {
        assert amount > 0;
        if (amount > this.balance) {
            return false; // insufficient funds
        }
        if (!receivingAccount.depositFunds(amount, "Transfer from " + this.name)) {
            return false; // could not add funds
        }
        this.withdrawFunds(amount, "Transfer to " + receivingAccount.name);
        return true;
    }

    /**
     * Called once at the end of each month to return a `String` summarizing the account's initial
     * balance that month, all transactions made during that month, and its final balance.
     */
    public final String transactionReport() {
        this.closeOutMonth();
        this.transactions.append("Final Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n");

        String report = this.transactions.toString();
        this.resetTransactionLog();
        return report;
    }

    /**
     * Performs any necessary end-of-month tasks for this account immediately before the transaction
     * report is finalized and returned to the client.
     */
    protected abstract void closeOutMonth();
}