package cs2110;

/**
 * Models a savings account in our personal finance app.
 */
public class SavingsAccount implements Account {

    /**
     * The name of this account.
     */
    private String name;

    /**
     * The current balance, in cents, of this account.
     */
    private int balance;

    /**
     * The current (nominal) APR of this account.
     */
    private double rate;

    /**
     * The current transaction report.
     */
    private StringBuilder transactions;

    /**
     * Constructs a savings account with the given `name`, initial `balance`, and interest `rate`.
     */
    public SavingsAccount(String name, int balance, double rate) {
        this.name = name;
        this.balance = balance;
        this.rate = rate;
        this.resetTransactionLog();
    }

    /**
     * Reassigns the transaction log to a new `StringBuilder` object that contains this month's
     * initial account balance
     */
    private void resetTransactionLog() {
        this.transactions = new StringBuilder("Initial Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n"); // newline
    }

    /**
     * Converts the given number of `cents` into a String in "$X.XX" format.
     */
    private static String centsToString(int cents) {
        int dollars = cents / 100;
        cents = cents % 100;
        return ("$" + dollars + "." + (cents < 10 ? "0" : "") + cents);
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public int balance() {
        return this.balance;
    }

    /**
     * Returns the current (nominal) APR of this account.
     */
    public double interestRate() {
        return this.rate;
    }

    /**
     * Deposits the specified `amount`, in cents, to the balance of the account, logs this
     * transaction with the given `memo`, and returns `true` to indicate that this deposit was
     * successful. Requires that `amount > 0`.
     */
    @Override
    public boolean depositFunds(int amount, String memo) {
        assert amount > 0;
        this.balance += amount;
        this.transactions.append(" - Deposit ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(": ");
        this.transactions.append(memo);
        this.transactions.append("\n");
        return true; // we can always add funds to a savings account
    }

    @Override
    public boolean transferFunds(Account receivingAccount, int amount) {
        assert amount > 0;
        if (amount > this.balance) {
            return false; // insufficient funds
        }
        if (!receivingAccount.depositFunds(amount, "Transfer from " + this.name)) {
            return false; // could not add funds
        }
        this.balance -= amount;
        this.transactions.append(" - Transfer ");
        this.transactions.append(centsToString(amount));
        this.transactions.append(" to ");
        this.transactions.append(receivingAccount.name());
        this.transactions.append("\n");
        return true;
    }

    /**
     * Called once at the end of each month to return a `String` summarizing the account's initial
     * balance that month, all transactions made during that month, and its final balance. As a
     * final transaction for the month, interest is accrued to the account based on the current
     * `rate`.
     */
    @Override
    public String transactionReport() {
        this.accrueMonthlyInterest();
        this.transactions.append("Final Balance: ");
        this.transactions.append(centsToString(this.balance));
        this.transactions.append("\n");

        String report = this.transactions.toString();
        this.resetTransactionLog();
        return report;
    }

    /**
     * Adds the monthly interest payment to the account balance.
     */
    private void accrueMonthlyInterest() {
        int interestAmount = (int) (this.balance * this.rate / (12 * 100));
        this.depositFunds(interestAmount, "Monthly interest @" + this.rate + "%");
    }
}
