/**
 * A portion of a task that must be performed in two parts, where all portions of the first
 * subtask must be completed before any work on the second subtask begins.
 */
class Task implements Runnable {
    /** Identifier for this portion of the task. */
    private int portion;

    /**
     * Barrier shared by all portions of the task; used to ensure that no work on the second
     * subtask begins until all work on the first subtask has finished.
     */
    private Barrier barrier;

    public Task(int portion, Barrier barrier) {
        this.portion = portion;
        this.barrier = barrier;
    }

    /**
     * Perform a portion of a task in two parts.  Part A will be completed for all portions before
     * any work on part B begins.
     */
    @Override
    public void run() {
        try {
            logStatus("Starting subtask A");
            randomSleep(500, 1000);
            logStatus("Finished subtask A");

            barrier.await();

            logStatus("Starting subtask B");
            randomSleep(500, 1000);
            logStatus("Finished subtask B");
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Print `message` to `System.out`, accompanied by this task's portion number.
     */
    void logStatus(String message) {
        System.out.println(message + " for portion " + portion);
    }

    /**
     * Sleep for a random number of milliseconds between minDuration
     * (inclusive) and maxDuration (exclusive).
     */
    static void randomSleep(int minDuration, int maxDuration) throws InterruptedException {
        java.util.Random rng = java.util.concurrent.ThreadLocalRandom.current();
        Thread.sleep(rng.nextInt(minDuration, maxDuration));
    }
}

public class BarrierDemo {
    public static void main(String[] args) {
        // Number of worker threads
        int nWorkers = 5;

        // Barrier to coordinate worker threads
        // Prints a message when all portions of the first subtask have been completed
        Barrier barrier = new Barrier(nWorkers,
                () -> System.out.println("\nSubtask A done\n"));

        // Create and start worker threads
        for (int i = 0; i < nWorkers; ++i) {
            Thread t = new Thread(new Task(i, barrier));
            t.start();
        }
    }
}
