import java.util.Iterator;

/**
 * Represents a vertex in a graph labeled with a string.
 */
public interface Vertex {
    /**
     * Return this vertex's label.
     */
    String label();

    /**
     * Return an object supporting iteration over all of the edges connecting this vertex to another
     * vertex in the graph.  This vertex serves as the "source" vertex for each such edge.
     */
    Iterable<Edge> outgoingEdges();

    /**
     * Return the number of edges connecting this vertex to another vertex in the graph.
     */
    int outDegree();

    /**
     * Return an object supporting iteration over all of the vertices adjacent to this vertex.
     */
    default Iterable<Vertex> successors() {
        // This just wraps `outgoingEdges()` so that its iterators yield each edge's destination
        // vertex rather than the edge itself.  (Annoyingly, a lot of Java boilerplate involving
        // anonymous classes is required for this simple transformation.)
        return new Iterable<Vertex>() {
            public Iterator<Vertex> iterator() {
                Iterator<Edge> outgoing = outgoingEdges().iterator();
                return new Iterator<>() {
                    public boolean hasNext() {
                        return outgoing.hasNext();
                    }
                    public Vertex next() {
                        return outgoing.next().destination();
                    }
                };
            }
        };
    }
}
