package cs2110;

import java.awt.Color;
import java.awt.GridLayout;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * Holds the grid of buttons that form a tic-tac-toe board.
 */
public class TicTacToeGrid extends JPanel {

    /**
     * Represents the current state of the TicTacToe game
     */
    private TicTacToe model;

    /**
     * The grid of buttons in each cell of this TicTacToe grid
     */
    private final TicTacToeCell[][] buttons;

    /**
     * Construct a JPanel with a 3 x 3 GridLayout of TicTacToeCell buttons.
     */
    public TicTacToeGrid() {
        model = new TicTacToe();

        buttons = new TicTacToeCell[3][3];

        setBackground(Color.BLACK);

        // A grid layout arranges components in a rectangular grid of equal-sized cells
        // The first two arguments are the dimensions of the grid (3 x 3)
        // The next two arguments (10 and 10) are the horizontal/vertical gaps between the cells
        setLayout(new GridLayout(3, 3, 10, 10));

        for (int i = 0; i < 3; i++) {
            for (int j = 0; j < 3; j++) {
                buttons[i][j] = new TicTacToeCell();
                buttons[i][j].addActionListener(cellAction(i,j));
                add(buttons[i][j]); // add button to grid layout
            }
        }

        repaint();
    }

    /**
     * Resets the game by generating a new model, clearing the symbols from all the cells, and
     * requesting a repaint.
     */
    public void reset() {
        model = new TicTacToe();
        for (int i = 0; i < 3; i++) {
            for (int j = 0; j < 3; j++) {
                buttons[i][j].reset();
            }
        }
        firePropertyChange("Turn", null, model.currentPlayer());
        repaint();
    }

    /**
     * Returns an ActionListener that responds to a user click in cell (i,j).
     */
    private ActionListener cellAction(int i, int j) {
        return e -> {
            char player = model.currentPlayer();

            // update model (changes model's current player)
            model.processMove(i, j);
            // update view
            buttons[i][j].addSymbol(player);
            // if game is over, show message dialog
            if (model.gameOver()) {
                String message = switch (model.winner()) {
                    case 'X', 'O' -> "Congratulations Player " + model.winner() + "!";
                    default -> "Tie game. Please play again!";
                };
                JOptionPane.showMessageDialog(this, message,
                        "Game Over", JOptionPane.PLAIN_MESSAGE);
                reset();
            }

            // alert any PropertyListeners that "Turn" has changed
            firePropertyChange("Turn", player, model.currentPlayer());
        };
    }
}