package cs2110;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;

/**
 * The primary class of our graphical Tic-Tac-Toe application.
 */
public class TicTacToeGraphical extends JFrame {

    /**
     * Constructs application window with title "Tic Tac Toe", a label displaying whose turn it is
     * at the top, a reset button at the bottom, and the game grid in the center.
     */
    public TicTacToeGraphical() {
        //////////////////////////////
        // View: Visible Components //
        //////////////////////////////

        setTitle("Tic-Tac-Toe");
        setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE); // gentle way to close application
        setResizable(false); // prevent resizing frame to make drawing code simpler

        TicTacToeGrid grid = new TicTacToeGrid();
        add(grid); // JFrame has default BorderLayout. This adds the grid to the center

        // create and add label for number of clicks to the top of the frame
        JLabel turnLabel = new JLabel("It's your turn Player X. Select a cell to claim.",
                SwingConstants.CENTER);
        turnLabel.setFont(turnLabel.getFont().deriveFont(16.0f)); // increase font size
        turnLabel.setBackground(Color.WHITE);
        turnLabel.setOpaque(true);
        turnLabel.setBorder(new EmptyBorder(10, 10, 10, 10)); // padding around text
        add(turnLabel, BorderLayout.NORTH);

        // create and add reset button to bottom of frame
        JButton resetButton = new JButton("Reset");
        resetButton.setFont(resetButton.getFont().deriveFont(20.0f)); // increase font size
        add(resetButton, BorderLayout.SOUTH);

        pack(); // adjust size of frame based on its components

        ////////////////////////////
        // Controller: Game Logic //
        ////////////////////////////

        // When the reset button is clicked, the GameGrid should be reset.
        resetButton.addActionListener(e -> grid.reset());
        // Reset Key Listener implemented as an anonymous class (with variable capture)
        // Cannot be implemented as a lambda expression because interface has 3 methods
        // Can use KeyAdapter instead to avoid having to override the methods we don't use
        resetButton.addKeyListener(new KeyListener() {
            @Override
            public void keyTyped(KeyEvent e) {
                if (e.getKeyChar() == 'r') {
                    grid.reset();
                }
            }

            @Override
            public void keyPressed(KeyEvent e) {
            }

            @Override
            public void keyReleased(KeyEvent e) {
            }
        });

        // An alternate way to add these listeners would be with private inner classes
        //resetButton.addActionListener(new ResetListener(grid));
        //resetButton.addKeyListener(new ResetKeyListener(grid));

        // When a player moves, update the turnLabel to say whose turn it is now.
        // The turnLabel is here, but turn state is in the grid, so we need
        // a PropertyChangeListener to detect the change in the model.
        grid.addPropertyChangeListener("Turn",
                e -> turnLabel.setText("It's your turn Player "
                        + e.getNewValue() + ". Select a cell to claim."));
    }
    /*
    // Inner class that serves as a key listener to reset the grid
    // Does not have access to variable capture, so needs a grid field
    private class ResetKeyListener implements KeyListener {
        TicTacToeGrid grid;
        public ResetKeyListener(TicTacToeGrid grid) {
            this.grid = grid;
        }

        @Override
        public void keyTyped(KeyEvent e) {
            if (e.getKeyChar() == 'r') {
                grid.reset();
            }
        }
        @Override public void keyPressed(KeyEvent e) {}
        @Override public void keyReleased(KeyEvent e) {}
    }
    */
    /*
    // Inner class that serves as an action listener for the reset button
    // Does not have access to variable capture, so needs a grid field
    // Can be implemented with a lambda expression because interface has only one method
    private class ResetListener implements ActionListener {
        TicTacToeGrid grid;  // The grid we want to be able to reset
        ResetListener(TicTacToeGrid grid) {// Constructs a reset listener with the given grid
            this.grid = grid;
        }
        // Resets the grid
        @Override
        public void actionPerformed(ActionEvent e) {
            grid.reset();
        }
    }
    */

    /**
     * Start the application
     */
    public static void main(String[] args) {
        // Creation of JFrame must occur on Event Dispatch Thread.
        SwingUtilities.invokeLater(() -> {
            TicTacToeGraphical game = new TicTacToeGraphical();
            game.setLocation(800, -700); // for external monitor above primary monitor
            game.setVisible(true);
        });
    }

}
