package cs2110;

public class TicTacToe {

    /**
     * The contents of the game grid, listed in row-major order. Includes exactly 9 entries that are
     * either 'X', 'O', or ' ' (indicating an empty cell).
     */
    private final char[] grid;
    /* Note: we've marked this field as final because we never reassign `grid` to reference a
     * different array than it is initialized to. We'll discuss final variables and immutability
     * more in an upcoming lecture.
     */

    /**
     * Whether the game is over, meaning a player has won or the entire board has been filled
     * without a winner, resulting in a tie.
     */
    private boolean gameOver;

    /**
     * The next player who will move. Must be 'X' or 'O' if `gameOver` is false, otherwise
     * unspecified.
     */
    private char currentPlayer;

    /**
     * The winner of the game. Must be 'X', 'O', or 'T' (to signify a tie) if `gameOver` is true,
     * otherwise unspecified.
     */
    private char winner;

    /**
     * Asserts that the `TicTacToe` class invariant holds
     */
    private void assertInv() {
        // `grid` includes 9 characters, each either 'X', 'O', or ' '
        assert grid != null && grid.length == 9;
        for (int i = 0; i < 9; i++) {
            assert grid[i] == 'X' || grid[i] == 'O' || grid[i] == ' ';
        }

        // `currentPlayer` is 'X' or 'O' when `gameOver` is false
        assert gameOver || currentPlayer == 'X' || currentPlayer == 'O';

        // `winner` is 'X', 'O', or 'T' when `gameOver` is true
        assert !gameOver || winner == 'X' || winner == 'O' || winner == 'T';
    }

    /**
     * Constructs a new TicTacToe game instance with an initially empty board and with 'X' as the
     * starting player.
     */
    public TicTacToe() {
        grid = new char[]{' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '};
        gameOver = false;
        currentPlayer = 'X';
        assertInv();
    }

    /**
     * Returns the character in position (`row`,`col`) of the grid, where ' ' indicates an empty
     * cell. Requires that `0 <= row <= 2` and `0 <= col <= 2`.
     */
    public char contentsOf(int row, int col) {
        assert 0 <= row && row <= 2 && 0 <= col && col <= 2; // defensive programming
        return grid[3 * row + col];
    }

    /**
     * Returns whether this game is over.
     */
    public boolean gameOver() {
        return gameOver;
    }

    /**
     * Returns the symbol of the next player to make a move. Requires that the game is not over.
     */
    public char currentPlayer() {
        assert !gameOver;
        return currentPlayer;
    }

    /**
     * Returns the symbol of the winning player, or 'T' if the game ended in a tie. Requires that
     * the game is over.
     */
    public char winner() {
        assert gameOver;
        return winner;
    }

    /**
     * Updates the game state to reflect a move by the `currentPlayer` at board position
     * (`row`,`col`). Requires that `0 <= row <= 2`, `0 <= col <= 2`, `contentsOf(row, col) == '
     * '`, and the game is not over.
     */
    public void processMove(int row, int col) {
        assert 0 <= row && row <= 2 && 0 <= col && col <= 2; // defensive programming
        assert contentsOf(row, col) == ' ';
        assert !gameOver;

        grid[3 * row + col] = currentPlayer;
        currentPlayer = (currentPlayer == 'X') ? 'O' : 'X';
        checkForGameOver(row, col);
        assertInv();
    }

    /**
     * Checks whether the most recent move at position (row, col) ended the game and updates the
     * values of `gameOver` and `winner` accordingly.
     */
    private void checkForGameOver(int row, int col) {
        // check for horizontal win
        if (grid[3 * row] == grid[3 * row + 1]
                && grid[3 * row] == grid[3 * row + 2]) {
            gameOver = true;
            winner = grid[3 * row];
            return;
        }

        // check for vertical win
        if (grid[col] == grid[3 + col] && grid[col] == grid[6 + col]) {
            gameOver = true;
            winner = grid[col];
            return;
        }

        // check for \ diagonal win
        if (row == col && grid[0] == grid[4] && grid[0] == grid[8]) {
            gameOver = true;
            winner = grid[0];
            return;
        }

        // check for / diagonal win
        if (row + col == 2 && grid[2] == grid[4] && grid[2] == grid[6]) {
            gameOver = true;
            winner = grid[2];
            return;
        }

        // check for tie game
        for (int i = 0; i < 9; i++) {
            if (grid[i] == ' ') {
                return; // still more empty cells
            }
        }
        // no more empty cells
        gameOver = true;
        winner = 'T';
    }
}