package cs2110;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

public class BFSDemo {

    /**
     * Uses a queue to carry out a BFS traversal of the vertices reachable from `source` in its
     * graph, building and returning a map that associates each discovered vertex with its level.
     */
    public static <V extends Vertex<E>,E extends Edge<V>> Map<String, Integer> bfsLevels(V source) {

        // Queue of discovered vertices that have not yet been visited
        Queue<V> frontier = new LinkedList<>();

        // Map associating each discovered vertex with its level
        Map<String, Integer> discovered = new LinkedHashMap<>();

        discovered.put(source.label(), 0);
        frontier.add(source);

        while(!frontier.isEmpty()) {
            V v = frontier.remove();
            for (E edge : v.outgoingEdges()) { // enqueue unvisited neighbors
                V neighbor = edge.head();
                if (!discovered.containsKey(neighbor.label())) {
                    discovered.put(neighbor.label(), discovered.get(v.label()) + 1);
                    frontier.add(neighbor);
                }
            }
        }

        return discovered;
    }

    /**
     * Auxiliary vertex properties relevant to unweighted shortest paths, the `level` of this vertex
     * in the BFS from the given `source` vertex, and the label of the `prev` vertex that had the
     * outgoing edge that discovered this vertex.
     */
    public record PathInfo(int level, String prev) {
        @Override
        public String toString() {
            return "{level = " + level + ", prev = " + prev + "}";
        }
    }

    /**
     * Uses a queue to carry out a BFS traversal of the vertices reachable from `source` in its
     * graph, building and returning a map that associates each discovered vertex with its level
     * in the search and the `prev` vertex that enabled its discovery.
     */
    public static <V extends Vertex<E>,E extends Edge<V>> Map<String, PathInfo> bfsPaths(V source) {

        // Queue of discovered vertices that have not yet been visited
        Queue<V> frontier = new LinkedList<>();

        // Map associating PathInfo to all discovered vertices
        Map<String, PathInfo> discovered = new LinkedHashMap<>();

        discovered.put(source.label(), new PathInfo(0, null)); // s does not have a "prev" vertex
        frontier.add(source);

        while(!frontier.isEmpty()) {
            V v = frontier.remove();

            for (E edge : v.outgoingEdges()) { // enqueue unvisited neighbors
                V neighbor = edge.head();
                if (!discovered.containsKey(neighbor.label())) {
                    int level = discovered.get(v.label()).level + 1;
                    discovered.put(neighbor.label(), new PathInfo(level, v.label()));
                    frontier.add(neighbor);
                }
            }
        }

        return discovered;
    }

    /**
     * Reconstructs and returns the shortest path from the vertex with label `srcLabel` to the
     * vertex with label `dstLabel` using the given `info` map produced by BFS.
     */
    public static List<String> reconstructPath(Map<String,PathInfo> info, String srcLabel, String dstLabel) {
        List<String> path = new LinkedList<>();
        path.add(dstLabel);
        while (!path.getFirst().equals(srcLabel)) {
            path.addFirst(info.get(path.getFirst()).prev());
        }
        return path;
    }

    /**
     * An unweighted, directed graph edge.
     */
    record UnweightedEdge(AdjListVertex<UnweightedEdge> tail, AdjListVertex<UnweightedEdge> head)
            implements Edge<AdjListVertex<UnweightedEdge>> {

    }

    /**
     * Constructs a new unweighted edge from the vertex with label `tailLabel` to the vertex with
     * label `headLabel` in `graph`. Throws an `IllegalArgumentException` if one of these endpoint
     * vertices does not exist in the graph, or if there is already an edge between them.
     */
    private static UnweightedEdge makeEdge(AdjListGraph<UnweightedEdge> graph, String tailLabel,
            String headLabel) {
        if (!graph.hasVertex(tailLabel)) {
            throw new IllegalArgumentException("No vertex labeled \"" + tailLabel + "\" in this graph.");
        }
        if (!graph.hasVertex(headLabel)) {
            throw new IllegalArgumentException("No vertex labeled \"" + headLabel + "\" in this graph.");
        }
        if (graph.hasEdge(tailLabel, headLabel)) {
            throw new IllegalArgumentException("Graph already has edge from \"" + tailLabel
                    + "\" to \"" + headLabel + "\".");
        }
        return new UnweightedEdge(graph.getVertex(tailLabel), graph.getVertex(headLabel));
    }

    /**
     * Returns the sequence of vertices in the given `path`, separated by " -> ".
     */
    public static String printPath(List<String> path) {
        StringBuilder sb = new StringBuilder();
        Iterator<String> it = path.iterator();
        while (it.hasNext()) {
            sb.append(it.next());
            if (it.hasNext()) {
                sb.append(" -> ");
            }
        }
        return sb.toString();
    }

    public static void main(String[] args) {
        AdjListGraph<UnweightedEdge> graph = new AdjListGraph<>();

        graph.addVertex("s");
        graph.addVertex("a");
        graph.addVertex("b");
        graph.addVertex("c");
        graph.addVertex("d");
        graph.addVertex("t");
        graph.addEdge(makeEdge(graph,"s","a"));
        graph.addEdge(makeEdge(graph,"s","b"));
        graph.addEdge(makeEdge(graph,"a","c"));
        graph.addEdge(makeEdge(graph,"b","a"));
        graph.addEdge(makeEdge(graph,"b","c"));
        graph.addEdge(makeEdge(graph,"c","d"));
        graph.addEdge(makeEdge(graph,"c","t"));
        graph.addEdge(makeEdge(graph,"d","b"));
        graph.addEdge(makeEdge(graph,"d","t"));

        System.out.println("Levels:");
        Map<String, Integer> map1 = bfsLevels(graph.getVertex("s"));
        for(String vLabel : map1.keySet()) {
            System.out.println(vLabel + ": " + map1.get(vLabel));
        }

        System.out.println("\nShortest Paths:");
        Map<String, PathInfo> map2 = bfsPaths(graph.getVertex("s"));
        for(String vLabel : map2.keySet()) {
            System.out.println(vLabel + ": " + map2.get(vLabel));
            System.out.println("Shortest Path: " + printPath(reconstructPath(map2, "s", vLabel)));
        }
    }
}