package cs2110;

import java.util.*;
import java.util.function.Consumer;

public class Traversals {

    /**
     * Uses a recursive helper method to carry out a depth-first search for the given `destination`
     * vertex starting from the given `source` vertex. Returns whether this search was successful
     */
    public static <V extends Vertex<E>,E extends Edge<V>> boolean dfs(V source, V destination) {
        Set<String> discovered = new HashSet<>();
        discovered.add(source.label());
        return dfsRecursive(source, destination, discovered);
    }

    /**
     * The recursive helper method for our depth-first search. Returns whether the `current` vertex
     * has a path to the `destination`.
     */
    public static <V extends Vertex<E>,E extends Edge<V>> boolean dfsRecursive(
            V current, V destination, Set<String> discovered) {

        if (current == destination) { // base case, we've reached the destination
            return true;
        }

        for (E edge : current.outgoingEdges()) { // complete visit by "discovering" all neighbors
            V neighbor = edge.head();
            if (!discovered.contains(neighbor.label())) { // neighbor hasn't been discovered yet
                discovered.add(neighbor.label()); // discover it
                if (dfsRecursive(neighbor, destination, discovered)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Carries out a DFS traversal of the vertices reachable from `source` in its graph.
     * Performs `preAction.accept()` when a vertex is first visited and performs
     * `postAction.accept()` when a vertex is settled.
     */
    public static <V extends Vertex<E>,E extends Edge<V>> void dfsTraverse(V source,
            Consumer<String> preAction, Consumer<String> postAction) {
        Set<String> discovered = new HashSet<>();
        discovered.add(source.label());
        dfsVisit(source, preAction, postAction, discovered);
    }

    /**
     * Traverses all *undiscovered* vertices reachable from `v` using a DFS, performing
     * `preAction.accept()` on each one as it is first visited and `postAction.accept()`
     * on each one as it is settled. `discovered` must contain labels of all discovered
     * vertices and will be added to as new vertices are discovered.
     */
    private static <V extends Vertex<E>,E extends Edge<V>> void dfsVisit(V v, Consumer<String>
                    preAction, Consumer<String> postAction, Set<String> discovered) {

        preAction.accept(v.label()); // start of v visit, perform preAction

        for (E edge : v.outgoingEdges()) { // complete visit by "discovering" all neighbors
            V neighbor = edge.head();
            if (!discovered.contains(neighbor.label())) { // neighbor hasn't been discovered yet
                discovered.add(neighbor.label()); // discover it
                dfsVisit(neighbor, preAction, postAction, discovered); // visit it
            }
        }

        postAction.accept(v.label()); // v is settled now, perform postAction
    }

    /**
     * Uses a queue to carry out a BFS traversal of the vertices reachable from `source` in its
     * graph. Performs `action.accept()` when a vertex is first visited.
     */
    public static <V extends Vertex<E>,E extends Edge<V>> void bfsQueue(V source,
            Consumer<String> action) {

        // Set of discovered vertices
        Set<String> discovered = new HashSet<>();

        // Queue of discovered vertices that have not yet been visited
        Queue<V> frontier = new LinkedList<>();

        discovered.add(source.label());
        frontier.add(source);

        while(!frontier.isEmpty()) {
            V v = frontier.remove();

            action.accept(v.label()); // start of v visit, perform action

            for (E edge : v.outgoingEdges()) { // enqueue unvisited neighbors
                V neighbor = edge.head();
                if (!discovered.contains(neighbor.label())) {
                    discovered.add(neighbor.label());
                    frontier.add(neighbor);
                }
            }
        }
    }
}