package cs2110;

import cs2110.AdjListGraph.AdjListVertex;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;

/**
 * Represents an unweighted directed graph using an adjacency list.
 */
public class AdjListGraph implements Graph<AdjListVertex, AdjListGraph.AdjListEdge> {

    /**
     * Represents an unweighted edge in this directed graph.
     */
    public record AdjListEdge(AdjListVertex tail, AdjListVertex head) implements Edge<AdjListVertex> {

    }

    /**
     * Represents a vertex in this graph responsible for tracking its neighbors.
     */
    public static class AdjListVertex implements Vertex<AdjListEdge> {

        /**
         * The label of this vertex
         */
        String label;

        /**
         * A map associating the labels of this vertex's neighbors with the edges connecting to
         * them.
         */
        LinkedHashMap<String, AdjListEdge> outEdges;

        /**
         * Constructs a new vertex with the given `label`.
         */
        AdjListVertex(String label) {
            this.label = label;
            outEdges = new LinkedHashMap<>();
        }

        @Override
        public String label() {
            return label;
        }

        @Override
        public int degree() {
            return outEdges.size();
        }

        @Override
        public boolean hasNeighbor(String headLabel) {
            return outEdges.containsKey(headLabel);
        }

        @Override
        public AdjListEdge edgeTo(String headLabel) {
            assert hasNeighbor(headLabel);
            return outEdges.get(headLabel);
        }

        @Override
        public Iterable<AdjListEdge> outgoingEdges() {
            return outEdges.values();
        }
    }

    /**
     * A map associating the labels of the vertices with their AdjListVertex objects.
     */
    HashMap<String, AdjListVertex> vertices;

    /**
     * Constructs a new graph initially containing no vertices.
     */
    public AdjListGraph() {
        vertices = new HashMap<>();
    }

    @Override
    public int vertexCount() {
        return vertices.size();
    }

    @Override
    public int edgeCount() {
        int count = 0;
        for (AdjListVertex v : vertices()) {
            count += v.degree();
        }
        return count;
    }

    @Override
    public boolean hasVertex(String label) {
        return vertices.containsKey(label);
    }

    @Override
    public AdjListVertex getVertex(String label) {
        assert hasVertex(label); // defensive programming
        return vertices.get(label);
    }

    @Override
    public boolean hasEdge(String tailLabel, String headLabel) {
        return vertices.containsKey(tailLabel) && vertices.get(tailLabel).hasNeighbor(headLabel);
    }

    @Override
    public AdjListEdge getEdge(String tailLabel, String headLabel) {
        assert hasEdge(tailLabel, headLabel); // defensive programming
        return vertices.get(tailLabel).edgeTo(headLabel);
    }

    @Override
    public void addVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException("Graph already contains vertex " + label);
        }
        vertices.put(label, new AdjListVertex(label));
    }

    @Override
    public AdjListEdge addEdge(String tailLabel, String headLabel) {
        if (!vertices.containsKey(tailLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + tailLabel);
        }
        AdjListVertex tail = vertices.get(tailLabel);

        if (!vertices.containsKey(headLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + headLabel);
        }
        AdjListVertex head = vertices.get(headLabel);

        if (tail.hasNeighbor(headLabel)) {
            throw new IllegalArgumentException("Graph already has edge from " + tailLabel + " to "
                    + headLabel);
        }

        AdjListEdge newEdge = new AdjListEdge(tail, head);
        tail.outEdges.put(headLabel, newEdge);
        return newEdge;
    }

    @Override
    public Iterable<AdjListVertex> vertices() {
        return vertices.values();
    }

    @Override
    public Iterable<AdjListEdge> edges() {
        ArrayList<AdjListEdge> edges = new ArrayList<>();
        for (AdjListVertex v : vertices()) {
            for (AdjListEdge e : v.outgoingEdges()) {
                edges.add(e);
            }
        }
        return edges;
    }

    @Override
    public String toString() {
        StringBuilder output = new StringBuilder();
        for (AdjListVertex v : vertices()) {
            output.append(v.label).append("-> [");
            for (AdjListEdge e : v.outgoingEdges()) {
                output.append(e.head().label()).append(",");
            }
            output.append("\b]\n");
        }
        return output.toString();
    }
}