package cs2110;

import cs2110.AdjListGraph.AdjListVertex;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * Represents a directed graph with int-weighted edges using an adjacency list.
 */
public class AdjListGraph implements Graph<AdjListVertex, WeightedEdge<AdjListVertex>> {

    /**
     * Represents a vertex in this graph responsible for tracking its neighbors.
     */
    public static class AdjListVertex implements Vertex<WeightedEdge<AdjListVertex>> {

        /**
         * The label of this vertex
         */
        String label;

        /**
         * A map associating the labels of this vertex's neighbors with the edges connecting to
         * them.
         */
        HashMap<String, WeightedEdge<AdjListVertex>> outEdges;

        AdjListVertex(String label) {
            this.label = label;
            outEdges = new HashMap<>();
        }

        @Override
        public String label() {
            return label;
        }

        @Override
        public int degree() {
            return outEdges.size();
        }

        @Override
        public boolean hasNeighbor(String headLabel) {
            return outEdges.containsKey(headLabel);
        }

        @Override
        public WeightedEdge<AdjListVertex> edgeTo(String headLabel) {
            assert hasNeighbor(headLabel);
            return outEdges.get(headLabel);
        }

        @Override
        public Iterable<WeightedEdge<AdjListVertex>> outgoingEdges() {
            return outEdges.values();
        }
    }

    /**
     * A map associating the labels of the vertices with their AdjListVertex objects.
     */
    HashMap<String, AdjListVertex> vertices;

    public AdjListGraph() {
        vertices = new HashMap<>();
    }

    @Override
    public int vertexCount() {
        return vertices.size();
    }

    @Override
    public int edgeCount() {
        int count = 0;
        for (AdjListVertex v : vertices()) {
            count += v.degree();
        }
        return count;
    }

    @Override
    public boolean hasVertex(String label) {
        return vertices.containsKey(label);
    }

    @Override
    public AdjListVertex getVertex(String label) {
        assert hasVertex(label); // defensive programming
        return vertices.get(label);
    }

    @Override
    public boolean hasEdge(String tailLabel, String headLabel) {
        return vertices.containsKey(tailLabel) && vertices.get(tailLabel).hasNeighbor(headLabel);
    }

    @Override
    public WeightedEdge<AdjListVertex> getEdge(String tailLabel, String headLabel) {
        assert hasEdge(tailLabel, headLabel); // defensive programming
        return vertices.get(tailLabel).edgeTo(headLabel);
    }

    @Override
    public void addVertex(String label) {
        if (hasVertex(label)) {
            throw new IllegalArgumentException("Graph already contains vertex " + label);
        }
        vertices.put(label, new AdjListVertex(label));
    }

    @Override
    public WeightedEdge<AdjListVertex> addEdge(String tailLabel, String headLabel) {
        if (!vertices.containsKey(tailLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + tailLabel);
        }
        AdjListVertex tail = vertices.get(tailLabel);

        if (!vertices.containsKey(headLabel)) {
            throw new IllegalArgumentException("Graph does not have a vertex " + headLabel);
        }
        AdjListVertex head = vertices.get(headLabel);

        if (tail.hasNeighbor(headLabel)) {
            throw new IllegalArgumentException("Graph already has edge from " + tailLabel + " to "
                    + headLabel);
        }

        WeightedEdge<AdjListVertex> newEdge = new WeightedEdge<>(tail, head);
        tail.outEdges.put(headLabel, newEdge);
        return newEdge;
    }

    @Override
    public Iterable<AdjListVertex> vertices() {
        return vertices.values();
    }

    @Override
    public Iterable<WeightedEdge<AdjListVertex>> edges() {
        ArrayList<WeightedEdge<AdjListVertex>> edges = new ArrayList<>();
        for (AdjListVertex v : vertices()) {
            for (WeightedEdge<AdjListVertex> e : v.outgoingEdges()) {
                edges.add(e);
            }
        }
        return edges;
    }

    @Override
    public String toString() {
        StringBuilder output = new StringBuilder();
        for (AdjListVertex v : vertices()) {
            output.append(v.label).append("-> [");
            for (WeightedEdge<AdjListVertex> e : v.outgoingEdges()) {
                output.append(e.head().label()).append("(w=").append(e.weight()).append("),");
            }
            output.append("\b]\n");
        }
        return output.toString();
    }
}
