package cs2110;

import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Random;

/**
 * A Set implementation backed by a chaining hash table
 */
public class HashSet<T> implements Set<T> {

    /**
     * The backing storage of this HashSet.
     */
    private LinkedList<T>[] buckets;

    /**
     * The number of elements stored in this HashSet.
     */
    private int size;

    /**
     * The initial capacity of the hash table.
     */
    public static final int INITIAL_CAPACITY = 5;

    /**
     * The maximum load factor permissible before resizing.
     */
    public static final double MAX_LOAD_FACTOR = 0.75;

    /**
     * Construct a new, initially empty, hash set.
     */
    public HashSet() {
        buckets = emptyTable(INITIAL_CAPACITY);
        size = 0;
    }

    /**
     * Constructs and returns an empty chaining hash table consisting with the given `capacity`.
     */
    private LinkedList<T>[] emptyTable(int capacity) {
        LinkedList<T>[] table = new LinkedList[capacity];
        for (int i = 0; i < capacity; i++) {
            table[i] = new LinkedList<>();
        }
        return table;
    }

    /**
     * Returns the hash value of the given `elem`
     */
    private int index(T elem) {
        return Math.abs(elem.hashCode() % buckets.length);
    }

    /**
     * Reassigns `buckets` to an array with double the capacity and re-hashes all entries.
     */
    private void doubleCapacity() {
        LinkedList<T>[] oldBuckets = buckets;
        buckets = emptyTable(buckets.length * 2);
        for (LinkedList<T> bucket : oldBuckets) {
            for (T elem : bucket) {
                buckets[index(elem)].add(elem);
            }
        }
    }

    @Override
    public boolean add(T elem) {
        assert elem != null;
        if (contains(elem)) {
            return false;
        }
        if ((double) (size + 1) / buckets.length > MAX_LOAD_FACTOR) { // exceed max load factor
            doubleCapacity();
        }
        buckets[index(elem)].add(elem);
        size += 1;
        return true;
    }

    @Override
    public boolean contains(T elem) {
        assert elem != null;
        return buckets[index(elem)].contains(elem);
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean remove(T elem) {
        assert elem != null;
        if (!contains(elem)) {
            return false;
        }
        buckets[index(elem)].remove(elem);
        size -= 1;
        return true;
    }

    @Override
    public Iterator<T> iterator() {
        return new HashSetIterator();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String[][] elements = new String[maxChainLength()][buckets.length];

        for (int i = 0; i < buckets.length; i++) {
            String[] contents = bucketString(i);
            for (int j = 0; j < contents.length; j++) {
                elements[j][i] = contents[j];
            }
        }

        for (int j = maxChainLength() - 1; j >= 0; j--) {
            for (int i = 0; i < buckets.length; i++) {
                sb.append("│");
                sb.append(elements[j][i]);
            }
            sb.append("│\n");
        }
        sb.append("└");
        for (int i = 0; i < buckets.length; i++) {
            sb.append("─".repeat(elements[0][i].length()));
            sb.append(i < buckets.length - 1 ? "┴" : "┘");
        }

        return sb.toString();
    }

    /**
     * Returns a String array of length `maxChainLength()` with entries corresponding to the
     * elements of `buckets[i]`, padded with spaces at the end so they all have the same length. Any
     * trailing array entries contains Strings of all spaces.
     */
    private String[] bucketString(int i) {
        int maxLength = 3;
        for (T elem : buckets[i]) {
            maxLength = Math.max(maxLength, elem.toString().length());
        }

        String[] contents = new String[maxChainLength()];
        Arrays.fill(contents, " ".repeat(maxLength));
        for (int j = 0; j < buckets[i].size(); j++) {
            contents[j] = buckets[i].get(j).toString();
            contents[j] = contents[j] + " ".repeat(maxLength - contents[j].length());
        }

        return contents;
    }

    /**
     * Returns the maximum number of elements stored in a bucket of this hash table.
     */
    private int maxChainLength() {
        int max = 0;
        for (LinkedList<T> bucket : buckets) {
            max = Math.max(max, bucket.size());
        }
        return max;
    }

    private class HashSetIterator implements Iterator<T> {

        /**
         * The index of the bucket containing the `next()` element.
         */
        private int current;

        /**
         * An iterator over the elements in the current bucket.
         */
        private Iterator<T> bucketIterator;

        public HashSetIterator() {
            current = 0;
            bucketIterator = buckets[current].iterator();
        }

        /**
         * Advances to the next bucket until locating one with an unseen element or running out of
         * buckets.
         */
        private void findNextElement() {
            while (!bucketIterator.hasNext() && current < buckets.length) {
                current += 1;
                if (current < buckets.length) {
                    bucketIterator = buckets[current].iterator();
                }
            }
        }

        @Override
        public boolean hasNext() {
            return current < buckets.length;
        }

        @Override
        public T next() {
            T elem = bucketIterator.next();
            findNextElement();
            return elem;
        }
    }
}
