package cs2110;

import java.util.ArrayList;

/**
 * A (binary) max heap storing elements of type T.
 */
public class MaxHeap<T extends Comparable<T>> {

    /**
     * The backing storage of this heap. Entries are ordered according to a level-order traversal of
     * a nearly complete binary tree and must satisfy the max heap order invariant.
     */
    private final ArrayList<T> heap;

    /**
     * Returns the index that would correspond to the parent of the heap entry at index `i`.
     * Requires that `i > 0` (the root node does not have a parent).
     */
    private static int parent(int i) {
        assert i > 0;
        return (i - 1) / 2;
    }

    /**
     * Returns the index that would correspond to the left child of the heap entry at index `i`.
     * Requires that `i >= 0`.
     */
    private static int leftChild(int i) {
        assert i >= 0;
        return 2 * i + 1;
    }

    /**
     * Returns the index that would correspond to the right child of the heap entry at index `i`.
     * Requires that `i >= 0`.
     */
    private static int rightChild(int i) {
        assert i >= 0;
        return 2 * i + 2;
    }

    /**
     * Asserts that this heap satisfies the max heap order invariant.
     */
    private void assertInv() {
        for (int i = 1; i < heap.size(); i++) {
            assert heap.get(i).compareTo(heap.get(parent(i))) <= 0;
        }
    }

    /**
     * Constructs an initially empty max heap.
     */
    public MaxHeap() {
        heap = new ArrayList<>();
        assertInv();
    }

    /**
     * Return the number of elements currently contained in this heap.
     */
    public int size() {
        return heap.size();
    }

    /**
     * Returns the largest element from this heap without removing it. Requires that the heap is not
     * empty.
     */
    public T peek() {
        assert size() > 0;
        return heap.getFirst();
    }

    /**
     * Adds the given `elem` to this heap.
     */
    public void add(T elem) {
        heap.add(elem);
        bubbleUp(heap.size() - 1);
        assertInv();
    }

    /**
     * Repeatedly swaps the element in index `i` of the heap with its parent until it resides at the
     * heap root or it is less than its new parent.
     */
    private void bubbleUp(int i) {
        if (i == 0) {
            return; // we've reached the root, no more swapping is necessary
        }
        int p = parent(i);
        if (heap.get(i).compareTo(heap.get(p)) > 0) { // parent smaller than `i`
            swap(i, p);
            bubbleUp(p); // tail-recursion, can be rewritten as while-loop to save space
        }
    }

    /**
     * Swaps the entries at indices `i` and `j` in this heap. Requires that `0 <= i < heap.size()`
     * and `0 <= j < heap.size(d)`.
     */
    private void swap(int i, int j) {
        T temp = heap.get(i);
        heap.set(i, heap.get(j));
        heap.set(j, temp);
    }

    /**
     * Removes and returns the largest element from this heap. Requires that the heap is not empty.
     */
    public T remove() {
        assert size() > 0;
        swap(0, heap.size() - 1);
        T removed = heap.removeLast();
        bubbleDown(0);
        assertInv();
        return removed;
    }

    /**
     * Repeatedly swaps the element in index `i` of this heap with its larger child until it resides
     * at a heap leaf or it is larger than all of its children.
     */
    private void bubbleDown(int i) {
        if (leftChild(i) >= heap.size()) {
            return; // `i` is a leaf
        }
        int c = leftChild(i); // index of larger child node
        if (rightChild(i) < heap.size() && heap.get(rightChild(i)).compareTo(heap.get(c)) > 0) {
            // `i` has a right child that is larger than its left child
            c = rightChild(i);
        }
        if (heap.get(i).compareTo(heap.get(c)) < 0) {
            swap(i, c);
            bubbleDown(c); // tail-recursion, can be rewritten as while-loop to save space
        }
    }

    /**
     * Recursively converts the array representation of this heap into a binary tree representation
     */
    private static <T> ImmutableBinaryTree<T> toTree(ArrayList<T> heap) {
        heap = new ArrayList<>(heap); // create a copy that we can safely mutate
        T root = heap.removeFirst();

        ArrayList<T> leftHeap = new ArrayList<>();
        ArrayList<T> rightHeap = new ArrayList<>();
        int rowSize = 1; // number of descendants in left subtree in current row;
        while (!heap.isEmpty()) {
            int r = rowSize;
            while (r > 0 && !heap.isEmpty()) {
                leftHeap.add(heap.removeFirst());
                r--;
            }
            r = rowSize;
            while (r > 0 && !heap.isEmpty()) {
                rightHeap.add(heap.removeFirst());
                r--;
            }
            rowSize *= 2;
        }

        return new ImmutableBinaryTree<>(root,
                leftHeap.isEmpty() ? null : toTree(leftHeap),
                rightHeap.isEmpty() ? null : toTree(rightHeap));
    }

    /**
     * Produces a multi-line String visualizing the tree structure of this max heap
     */
    @Override
    public String toString() {
        return toTree(heap).toString();
    }
}
