package cs2110;

public class Heapsort {

    /**
     * Sorts the entries of `a` by first turning them into a heap and then removing the entries
     * sequentially in descending sorted order.
     */
    public static <T extends Comparable<T>> void heapSort(T[] a) {
        heapify(a);

        /* Loop invariant: `a[..i)` is a valid binary max heap. `a[i..]` is sorted and `>= a[0]`. */
        for (int i = a.length; i > 0; i--) {
            assertHeap(a, i);
            assertSorted(a, i);
            remove(a, i);
        }
        assertSorted(a, 0); // post-condition
    }

    /**
     * Rearranges the entries of `a` so that they correspond to a valid binary max heap.
     */
    private static <T extends Comparable<T>> void heapify(T[] a) {
        /* Loop invariant: `a[..i)` is a valid binary max heap. */
        for (int i = 1; i < a.length; i++) {
            assertHeap(a, i);
            add(a, i);
        }
        assertHeap(a, a.length); // post-condition
    }

    /**
     * Asserts that `a[..i)` satisfies the max heap order invariant.
     */
    private static <T extends Comparable<T>> void assertHeap(T[] a, int i) {
        for (int j = 1; j < i; j++) {
            assert a[j].compareTo(a[(j - 1) / 2]) <= 0;
        }
    }

    /**
     * Asserts that `a[i..]` are sorted.
     */
    private static <T extends Comparable<T>> void assertSorted(T[] a, int i) {
        for (int j = i; j < a.length - 1; j++) {
            assert a[j].compareTo(a[j + 1]) <= 0;
        }
    }

    /**
     * Adds `a[i]` to the heap `a[..i)` by performing a `bubbleUp()` operation starting at index
     * `i`. Requires that `i < a.length`.
     */
    private static <T extends Comparable<T>> void add(T[] a, int i) {
        assert i < a.length;
        bubbleUp(a, i);
    }

    /**
     * Repeatedly swaps the element in index `i` of the heap with its parent until it resides at the
     * heap root or it is less than its new parent. Requires that `a[..i)` is a valid max heap.
     */
    private static <T extends Comparable<T>> void bubbleUp(T[] a, int i) {
        if (i == 0) {
            return; // we've reached the root, no more swapping is necessary
        }
        int p = (i - 1) / 2;
        if (a[i].compareTo(a[p]) > 0) { // parent smaller than `i`
            swap(a, i, p);
            bubbleUp(a, p); // tail-recursion, can be rewritten as while-loop to save space
        }
    }

    /**
     * Swaps the entries at indices `i` and `j` in this heap. Requires that `0 <= i < heap.size()`
     * and `0 <= j < heap.size(d)`.
     */
    private static <T extends Comparable<T>> void swap(T[] a, int i, int j) {
        T temp = a[i];
        a[i] = a[j];
        a[j] = temp;
    }

    /**
     * Relocates the maximum entry of the max heap `a[..i)` to `a[i-1]` and restores the heap order
     * invariant on `a[..i-1)`. Requires that `i > 0`.
     */
    public static <T extends Comparable<T>> void remove(T[] a, int i) {
        assert i > 0;
        swap(a, 0, i - 1);
        bubbleDown(a, 0, i - 1);
    }

    /**
     * Repeatedly swaps the element in index `i` of this heap with its larger child until it resides
     * at a heap leaf or it is larger than all of its children. Requires that `a[..j)` is a valid
     * max heap.
     */
    private static <T extends Comparable<T>> void bubbleDown(T[] a, int i, int j) {
        if (2 * i + 1 >= j) {
            return; // `i` is a leaf
        }
        int c = 2 * i + 1; // index of larger child node, initialize to left child
        if (c + 1 < j && a[c + 1].compareTo(a[c]) > 0) {
            // `i` has a right child that is larger than its left child
            c++;
        }
        if (a[i].compareTo(a[c]) < 0) {
            swap(a, i, c);
            bubbleDown(a, c, j); // tail-recursion, can be rewritten as while-loop to save space
        }
    }

    public static <T> String toString(T[] a) {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (int i = 0; i < a.length - 1; i++) {
            sb.append(a[i]);
            sb.append(",");
        }
        sb.append(a[a.length - 1]);
        sb.append("]");
        return sb.toString();
    }

    public static void main(String[] args) {
        Integer[] a = new Integer[]{1, 2, 8, 3, 5, 7, 2, 4, 5, 6, 8};
        heapSort(a);
        System.out.println(toString(a));
    }
}
