package cs2110;

import java.util.Comparator;
import java.util.Iterator;

/**
 * An unbalanced binary search tree that uses a comparator to establish its order invariant.
 */
public class BSTComparator<T> extends BinaryTree<T> {

    /**
     * The left subtree of this BST. All elements in `left` must be "<=" `root`. Must be `null` if
     * `root == null` and `!= null` if `root != null`.
     */
    protected BSTComparator<T> left;

    /**
     * The right subtree of this BST. All elements in `right` must be ">=" `root`. Must be `null` if
     * `root == null` and `!= null` if `root != null`.
     */
    protected BSTComparator<T> right;

    /**
     * Used to establish the order invariant by comparing elements of typee T.
     */
    protected Comparator<T> cmp;

    /**
     * Construct an empty BST.
     */
    public BSTComparator(Comparator<T> cmp) {
        this.cmp = cmp;
        root = null;
        left = null;
        right = null;
    }

    @Override
    protected BinaryTree<T> left() {
        return (left.root == null) ? null : left;
    }

    @Override
    protected BinaryTree<T> right() {
        return (right.root == null) ? null : right;
    }

    /**
     * Returns the number of elements stored in this BST. An empty BST stores 0 elements.
     */
    @Override
    public int size() {
        return (root == null) ? 0 : super.size();
    }

    /**
     * Returns the height of this BST. An empty BST has height 0.
     */
    @Override
    public int height() {
        return (root == null) ? 0 : super.height();
    }

    /**
     * Adds the given `elem` into this BST at a location that respects the BST order condition.
     * Requires that `elem != null`.
     */
    public void add(T elem) {
        assert elem != null;
        BSTComparator<T> loc = find(elem);
        if (loc.root != null) { // `elem` is already in the BST
            if (loc.right.root == null) {
                // `elem` doesn't have a right subtree; make new `elem` its right child.
                loc = loc.right;
            } else {
                // `elem` has a right subtree; make new `elem` the left child of its successor.
                loc = successorDescendant().left;
                /* This will result in a very unbalanced tree when lots of duplicate elements
                 * are added. An alternate (more complicated) approach can correct this. */
            }
        }
        loc.root = elem;
        loc.left = new BSTComparator<>(cmp);
        loc.right = new BSTComparator<>(cmp);
    }

    /**
     * Locates and returns a subtree whose root is `elem`, or the leaf child where `elem` would be
     * located if `elem` is not in this BST. Requires that `elem != null`.
     */
    private BSTComparator<T> find(T elem) {
        assert elem != null;
        if (root == null || cmp.compare(elem, root) == 0) {
            return this;
        } else if (cmp.compare(elem, root) < 0) { // `elem < root`, recurse left
            return left.find(elem);
        } else { // `elem > root`, recurse right
            return right.find(elem);
        }
    }

    /**
     * Returns the node in this subtree that comes immediately after its root in an in-order
     * traversal. Requires that `right.root != null`, so this BST has a non-empty right subtree.
     */
    private BSTComparator<T> successorDescendant() {
        assert right.root != null;
        BSTComparator<T> current = right;
        while (current.left.root != null) {
            current = current.left;
        }
        return current;
    }

    /**
     * Returns whether this BST contains `elem`. Requires that `elem != null`.
     */
    public boolean contains(T elem) {
        assert elem != null;
        return find(elem).root != null;
    }

    /**
     * Removes the given `elem` from this BST. Requires that `elem != null` and `contains(elem) ==
     * true`
     */
    public void remove(T elem) {
        assert contains(elem);
        BSTComparator<T> loc = find(elem);
        if (loc.right.root == null) { // `elem` has no right subtree
            loc.supplantWith(loc.left); // supplant it with its left subtree
        } else { // `elem` has a right subtree; replace it with its successor
            BSTComparator<T> successor = loc.successorDescendant();
            loc.root = successor.root; // copy up value in successor
            // successor does not have left subtree
            successor.supplantWith(successor.right); // supplant successor with its right subtree
        }
    }

    /**
     * Replace this tree with the given `other` tree.
     */
    private void supplantWith(BSTComparator<T> other) {
        root = other.root;
        left = other.left;
        right = other.right;
    }

    public static void main(String[] args) {
        BSTComparator<Integer> tree = new BSTComparator<>(Integer::compare);
        /* Note: The "::" notation on the previous line is called a method reference, which
         * is syntactic sugar for the lambda expression (int x, int y) -> Integer.compare(x,y).
         * Since `Comparator` is a functional interface, we can instantiate the constructor
         * argument using a lambda expression.
         */
        for (int i : new int[]{8, 4, 12, 2, 6, 10, 14, 1, 3, 5, 7, 9, 11, 13, 15}) {
            tree.add(i);
        }
        System.out.println(tree);

        tree.remove(8);
        System.out.println(tree);
        tree.remove(4);
        System.out.println(tree);
        tree.remove(9);
        System.out.println(tree);

        System.out.print("In-order: ");
        for (Integer e : tree) {
            System.out.print(e + " ");
        }

        System.out.print("\nPre-order: ");
        Iterator<Integer> it = tree.preorderIterator();
        while (it.hasNext()) {
            System.out.print(it.next() + " ");
        }

        System.out.print("\nPost-order: ");
        it = tree.postorderIterator();
        while (it.hasNext()) {
            System.out.print(it.next() + " ");
        }
    }
}
