package cs2110;

import java.util.Scanner;

public class ExpressionEvaluator {

    /**
     * Evaluates the given well-formed mathematical expression `expr` and returns its value. For
     * this basic version of the method (described in lecture), the operands must be single-digit
     * integers and the only supported operations are addition (+) and multiplication (*). The
     * allowable characters in the expression are, therefore, {0,1,2,3,4,5,6,7,8,9,+,*,(,)}.
     */
    public static int evaluate(String expr) {
        Stack<Integer> operands = new LinkedStack<>();
        Stack<Character> operators = new LinkedStack<>(); // invariant: contains only '(', '+', and '*'

        /*
         * Keeps track of our "state" as we are parsing `expr`, which helps us to detect and alert
         * the client about mal-formed expressions. There are two possible states:
         * (1) We have more-recently read a digit than an operator, the next character we read
         *     must be an operator or a ')'.
         * (2) We have more-recently read an operator than an operator, the next character we read
         *     must be a digit or a '('.
         * We can use a boolean to model this state, where `true` corresponds to state (1) and
         * `false` corresponds to state (2).
         */
        boolean expectingOperator = false; // in infix notation, the first operand comes before an operator

        for (char c : expr.toCharArray()) { // arrays are Iterable, so can be used in enhanced-for loops
            if (c == '(') {
                /*
                 * Below, we use a slightly different syntax for an `assert` statement than we have
                 * seen before. By adding a colon (:) after the boolean expression followed by a
                 * String, Java will include the String in the stack trace error message.
                 */
                assert !expectingOperator : "'(' cannot follow an operand";
                operators.push('(');
            } else if (c == '*') {
                assert expectingOperator : "'*' must follow an operand, not an operator";
                while (!operators.isEmpty() && operators.peek() == '*') {
                    oneStepSimplify(operands, operators);
                }
                operators.push('*');
                expectingOperator = false;
            } else if (c == '+') {
                assert expectingOperator : "'+' must follow an operand, not an operator";
                while (!operators.isEmpty() && (operators.peek() == '*'
                        || operators.peek() == '+')) {
                    oneStepSimplify(operands, operators);
                }
                operators.push('+');
                expectingOperator = false;
            } else if (c == ')') {
                assert expectingOperator : "')' must follow an operand, not an operator";
                assert !operators.isEmpty() : "mismatched parentheses, extra ')'";
                while (operators.peek() != '(') {
                    oneStepSimplify(operands, operators);
                    assert !operators.isEmpty() : "mismatched parentheses, extra ')'";
                }
                operators.pop(); // remove '('
            } else { // c is a digit
                assert c >= '0' && c <= '9' : "expression contains an illegal character";
                assert !expectingOperator : "digits cannot follow an operand, our evaluator only supports single-digit inputs";
                operands.push(c - '0'); // convert c to an int and auto-box
                expectingOperator = true;
            }
        }
        assert expectingOperator : "expression must end with an operand, not an operator";
        while (!operators.isEmpty()) {
            assert operators.peek() != '(' : "mismatched parentheses, extra '('";
            oneStepSimplify(operands, operators);
        }
        // If the above assertions pass, the operands stack should include exactly one value,
        // the return value. We'll include two assertions to verify this as a sanity check.
        assert !operands.isEmpty();
        int result = operands.pop();
        assert operands.isEmpty();
        return result;
    }

    /**
     * Helper method that partially simplifies the expression by `pop()`ping one operator from the
     * `operators` stack, `pop()`ping its two operands from the `operands` stack, evaluating the
     * operator, and then `push()`ing its result onto the `operands` stack. Requires that
     * `opererators.peek()` is '+' or '*' and `operands` includes at least two elements.
     */
    private static void oneStepSimplify(Stack<Integer> operands, Stack<Character> operators) {
        char op = operators.pop();
        assert op == '+' || op == '*';

        int o2 = operands.pop(); // second operand is higher on stack
        int o1 = operands.pop();
        operands.push(op == '+' ? o1 + o2 : o1 * o2);
    }

    /**
     * A very basic calculator application.
     */
    public static void main(String[] args) {
        try (Scanner in = new Scanner(System.in)) {
            while (true) { // repeat indefinitely
                System.out.print("Enter an expression, or enter \"q\" to quit: ");
                String expr = in.next();
                if (expr.equals("q")) {
                    break; // exit loop
                }
                System.out.println("= " + evaluate(expr));
            }
        }
    }
}
