package cs2110;

import java.util.Arrays;
import java.util.Iterator;

/**
 * An implementation of the CS2110List ADT using a dynamic array.
 */
public class DynamicArrayList<T> implements CS2110List<T> {

    /**
     * The backing storage of this list. Must have `storage[..size) != null` and `storage[size..) ==
     * null`.
     */
    private T[] storage;

    /**
     * The current size of this list. Must have `0 <= size <= storage.length`.
     */
    private int size;

    /**
     * The initial capacity for the backing storage array.
     */
    protected static final int INITIAL_CAPACITY = 10;

    /**
     * Asserts the DynamicArrayList class invariant.
     */
    private void assertInv() {
        assert storage != null; // implicit invariant
        assert 0 <= size;
        assert size <= storage.length;

        for (int i = 0; i < size; i++) {
            assert storage[i] != null;
        }
        for (int j = size; j < storage.length; j++) {
            assert storage[j] == null;
        }
    }

    /**
     * Constructs a new, initially empty, DynamicArrayList.
     */
    public DynamicArrayList() {
        size = 0;
        storage = (T[]) new Object[INITIAL_CAPACITY];
        assertInv();
    }

    /**
     * Copies the current entries of `storage` to a new backing array with double the capacity.
     */
    private void increaseCapacity() {
        storage = Arrays.copyOf(storage, 2 * storage.length);
    }

    @Override
    public void add(T elem) {
        insert(size, elem);
    }

    @Override
    public void insert(int index, T elem) {
        assert elem != null; // defensive programming
        assert 0 <= index;
        assert index <= size;
        if (size == storage.length) {
            increaseCapacity();
        }
        assert size < storage.length; // after potential resize
        System.arraycopy(storage, index, storage, index + 1, size - index); // right shift
        storage[index] = elem;
        size++;
        assertInv();
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public T get(int index) {
        assert index >= 0; // defensive programming
        assert index < size;
        return storage[index];
    }

    /**
     * Returns the index of the first instance of `elem` in this list. Returns `size` if `elem` is
     * not present in this list.
     */
    private int find(T elem) {
        for (int i = 0; i < size; i++) {
            if (storage[i].equals(elem)) {
                return i;
            }
        }
        return size;
    }

    @Override
    public boolean contains(T elem) {
        return find(elem) < size;
    }

    @Override
    public int indexOf(T elem) {
        int i = find(elem);
        assert i < size; // `elem` is present in list
        return i;
    }

    @Override
    public void set(int index, T elem) {
        assert elem != null; // defensive programming
        assert index >= 0;
        assert index < size;
        storage[index] = elem;
        assertInv();
    }

    @Override
    public T remove(int index) {
        assert 0 <= index;
        assert index < size;
        T removed = storage[index];
        System.arraycopy(storage, index + 1, storage, index, size - index - 1); // left shift
        size--;
        storage[size] = null; // restore class invariant
        assertInv();
        return removed;
    }

    @Override
    public void delete(T elem) {
        remove(find(elem)); // `remove()` call asserts pre-condition and class invariant
    }

    /**
     * Returns a String representation of this list, separating its elements with commas and
     * enclosing the list in square brackets [].
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (int i = 0; i < size; i++) {
            sb.append(storage[i]);
            if (i < size - 1) {
                sb.append(",");
            }
        }
        sb.append("]");
        return sb.toString();
    }

    @Override
    public Iterator<T> iterator() {
        return new DynamicArrayListIterator();
    }

    private class DynamicArrayListIterator implements Iterator<T> {

        /**
         * The index of the next element that will be returned by `next()`.
         */
        private int index;

        /**
         * Constructs a new iterator object beginning at the start of this list.
         */
        public DynamicArrayListIterator() {
            index = 0;
        }

        @Override
        public boolean hasNext() {
            return index < size;
        }

        @Override
        public T next() {
            T elem = storage[index];
            index++;
            return elem;
        }
    }
}
