package cs2110;

import java.awt.Color;
import java.awt.GridLayout;
import java.awt.event.ActionListener;
import java.util.Random;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * A panel containing a grid of Light buttons
 */
public class LightGrid extends JPanel {

    /**
     * Grid size (Model)
     */
    private final int size;

    /**
     * Number of clicks since the game was reset. (Model)
     */
    private int numClicks;

    /**
     * Number of lit cells. Used to check if game has been won. (Model)
     */
    private int numLit;

    /**
     * Random generator used to initialize random board states (Model)
     */
    private final Random rand;

    /**
     * Array of lightbulb buttons (View)
     */
    private final LightButton[][] buttons;

    /**
     * Construct a JPanel with a `size`x`size` GridLayout of lightbulb buttons
     * <p> Initialize the model to a random configuration of lights
     */
    public LightGrid(int size) {
        this.size = size;
        rand = new Random(12345); // seed the randomness for reproducibility during testing
        setBackground(Color.DARK_GRAY);

        buttons = new LightButton[this.size][this.size];
        // A grid layout arranges components in a rectangular grid of equal-sized cells
        // The first two arguments `size`x`size` are the dimensions of the grid
        // The next two arguments (10 and 10) are the horizontal/vertical gaps between the cells
        setLayout(new GridLayout(size, size, 10, 10));

        for (int i = 0; i < this.size; i++) {
            for (int j = 0; j < this.size; j++) {
                buttons[i][j] = new LightButton(); // create a new button
                buttons[i][j].addActionListener(processClick(i,j));
                add(buttons[i][j]); // add button to grid layout
            }
        }

        reset(); // set up the game instance
    }

    ////////////////////////////
    // Controller: Game Logic //
    ////////////////////////////

    /**
     * Resets the game by generating a new random board and setting the number of clicks to 0.
     */
    public void reset() {
        for (int i = 0; i < size; i++) {
            for (int j = 0; j < size; j++) {
                if (rand.nextBoolean()) {
                    toggleNeighborhood(i, j);
                }
            }
        }
        setNumClicks(0);
    }

    /**
     * Update the value of `numClicks` to `newClicks`. The value of `numClicks` should ONLY be
     * modified through this method to ensure that any observers are properly notified.
     */
    private void setNumClicks(int newNumClicks) {
        int oldNumClicks = numClicks;
        numClicks = newNumClicks;
        firePropertyChange("Clicks", oldNumClicks, numClicks);
    }

    /**
     * Process a click of cell (i,j) by toggling the surrounding cells, increasing the number of
     * clicks, and displaying a dialog if the player has won.
     */
    private ActionListener processClick(int i, int j) {
        return e -> {
            toggleNeighborhood(i, j);
            setNumClicks(numClicks + 1);

            if (numLit == 0) { // show win message
                JOptionPane.showMessageDialog(this, "You won with " + numClicks +
                                " clicks.", "Congratulations!",
                        JOptionPane.PLAIN_MESSAGE);
                reset(); // reset for a new game
            }
        };
    }

    /**
     * Toggles cell (i,j) and any of its neighboring cells, updates 'numLit' appropriately.
     */
    private void toggleNeighborhood(int i, int j) {
        toggleButton(i, j);
        if (i > 0) {
            toggleButton(i - 1, j);
        }
        if (i < size - 1) {
            toggleButton(i + 1, j);
        }
        if (j > 0) {
            toggleButton(i, j - 1);
        }
        if (j < size - 1) {
            toggleButton(i, j + 1);
        }
    }
    /**
     * Toggles button at position (i,j) and updates 'numLit'
     */
    private void toggleButton(int i, int j) {
        numLit += buttons[i][j].toggle();
    }
}