import java.util.Arrays;
import java.util.Comparator;
import java.util.Random;

/**
 * An element type for sorting demos.  Has a `value` property and an independent `id` property.
 */
class Element {
    final int value;
    final char id;

    Element(int value, char id) {
        this.value = value;
        this.id = id;
    }

    @Override
    public String toString() {
        return String.valueOf(value) + "." + id;
//        return String.valueOf(value);  // Use this line to only print what the Comparator sees
    }
}

/**
 * Order `Element`s by their `value` (ignoring their `id`).  Also keeps track of how many
 * comparisons it is used to make.
 */
class ValueComparison implements Comparator<Element> {
    /** Number of times `compare()` has been called since construction or last reset. */
    private int compareCount = 0;

    @Override
    public int compare(Element a, Element b) {
        compareCount += 1;
        return Integer.compare(a.value, b.value);
    }

    /** Reset comparison counter to 0. */
    public void reset() {
        compareCount = 0;
    }

    /** Return number of comparisons since construction or last reset. */
    public int compareCount() {
        return compareCount;
    }
}

/**
 * A functional interface for sorting algorithms (allows methods to take a client's choice of
 * algorithm as an _argument_).
 */
interface Sorter<T> {
    /**
     * Sort the elements of `a` in ascending order (as determined by `cmp`).
     */
    void sort(T[] a, Comparator<T> cmp);
}

public class SortingDemo {
    public static void main(String[] args) {
        // NEW SYNTAX: `ClassName::methodName`
        // This is called a "method reference"; you don't need to know how it works until we
        // discuss "lambda expressions" later in the course.
        // For now, know that it creates an object that implements the `Sorter` interface by calling
        // the specified methodName.  (Sorry, but it makes demo code like this so much shorter.)

        // DEMO: Stability
        showStability(Sorting::insertionSort);
        System.out.println();


        // DEMO: Compare performance
        System.out.println("## Selection sort");
        benchmarkSort((a, cmp) -> Sorting.selectionSort(a, cmp));
        System.out.println();

        System.out.println("## Insertion sort");
        benchmarkSort(Sorting::insertionSort);
        System.out.println();

        System.out.println("## Merge sort");
        benchmarkSort(Sorting::mergeSort);
        System.out.println();

        System.out.println("## Quicksort");
        benchmarkSort(Sorting::quickSort);
        System.out.println();
    }


    /**
     * Create and return a shuffled array of `length` Elements of arbitrary value and id.
     * Duplicate element "values" are guaranteed for `length > 1`; duplicate elemenbt "ids" are
     * guaranteed for `length > 2`.
     */
    static Element[] makeArray(int length) {
        // Start by generating a sorted array of pairs of elements with matching values but
        // different IDs ('a' and 'b').
        Element[] ans = new Element[length];
        for (int i = 0; i < length; i += 2) {
            ans[i] = new Element(i / 2, 'a');
            ans[i + 1] = new Element(i / 2, 'b');
        }

        // Shuffle the elements using the Fisher-Yates algorithm.
        // (the Random Number Generator seed is fixed for reproducibility)
        Random rng = new Random(1);
        for (int i = 0; i < length - 1; ++i) {
            int j = rng.nextInt(i, length);
            Sorting.swap(ans, i, j);
        }

        return ans;
    }

    /**
     * Print an array before and after sorting to visualize the stability of the selected algorithm.
     */
    static void showStability(Sorter<Element> sorter) {
        // Create shuffled array and a comparator to use for sorting
        Comparator<Element> cmp = new ValueComparison();
        Element[] a = makeArray(10);

        // Print shuffled array
        System.out.println(Arrays.toString(a));

        // Sort array
        sorter.sort(a, cmp);

        // Print sorted array
        System.out.println(Arrays.toString(a));
    }

    /**
     * Measure the performance of the selected sorting algorithm.
     */
    static void benchmarkSort(Sorter<Element> sorter) {
        // If true, fill array with duplicate values
        boolean duplicates = false;

        ValueComparison cmp = new ValueComparison();
        int length = 13000;
        Element[] a;
        Element dup = new Element(0, 'a');

        // Warmup JVM using many iterations of a short array
        if (duplicates) {
            a = new Element[100];
            Arrays.fill(a, dup);
        } else {
            a = makeArray(100);
        }
        for (int iter = 0; iter < 1000; ++iter) {
            sorter.sort(a, cmp);
        }

        // Measure performance on a full-length array
        if (duplicates) {
            a = new Element[length];
            Arrays.fill(a, dup);
        } else {
            a = makeArray(length);
        }
        cmp.reset();
        long startTime = System.nanoTime();
        sorter.sort(a, cmp);
        long endTime = System.nanoTime();
        System.out.println("Time: " + 1e-9*(endTime - startTime) + " s");
        System.out.println("Comparisons: " + cmp.compareCount());
    }
}