package cs2110;

import java.nio.BufferOverflowException;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * A buffer of values of type E with a fixed capacity.
 * This class is not thread-safe.
 */
public class RingBufferBQ<E> implements BoundedQueue<E> {

    /**
     * Backing array storing the values; length is buffer capacity. Elements that do not correspond
     * to contained values are null.
     */
    final E[] store;

    /**
     * Index of next value that will be returned by `get()`.
     */
    int iHead;

    /**
     * Number of values currently contained in buffer. Index of next element that can be written to
     * by `put()` is:
     *
     * @{code (iHead + size)%store.length}.
     */
    int size;

    /**
     * Returns true if class invariant is satisfied.
     */
    private boolean checkInvariant() {
        // Note: does not check that unoccupied elements are null
        return iHead >= 0 && iHead < store.length &&
                size >= 0 && size <= store.length;
    }

    /**
     * Create a new buffer with the specified capacity.
     */
    @SuppressWarnings("unchecked")
    public RingBufferBQ(int capacity) {
        store = (E[]) new Object[capacity];
        iHead = 0;
        size = 0;
        assert checkInvariant();
    }

    /**
     * Returns the total capacity of the buffer (the number of elements it contains when it is
     * full).
     */
    public int capacity() {
        return store.length;
    }

    /**
     * Returns true if no element can currently be added to the buffer.
     */
    public boolean isFull() {
        assert checkInvariant();
        return size == store.length;
    }

    /**
     * Returns true if no element can be consumed from the buffer.
     */
    public boolean isEmpty() {
        assert checkInvariant();
        return size == 0;
    }

    /**
     * Returns the number of elements currently in the buffer.
     */
    public int size() {
        assert checkInvariant();
        return size;
    }

    /**
     * Append a value `x` to the buffer. Throws BufferOverflowException if buffer is full.
     */
    public void put(E x) {
        assert checkInvariant();

        // TODO 2A: Write a spin loop that attempts to block until there is room
        //  in the buffer.

        if (isFull()) {
            throw new BufferOverflowException();
        }

        // Compute index of next available element
        int iEnd = (iHead + size) % store.length;
        assert store[iEnd] == null;

        // Store value
        store[iEnd] = x;
        size += 1;

        assert checkInvariant();
    }

    /**
     * Remove and return the oldest value in the buffer. Throws NoSuchElementException if buffer is
     * empty.
     */
    public E get() {
        assert checkInvariant();

        // TODO 2B: Write a spin loop that attempts to block until there is an
        //  element in the buffer.

        if (isEmpty()) {
            throw new NoSuchElementException();
        }

        // TODO 1: Implement get() according to specification. Call checkInvariant() before
        // returning to ensure that modifications did not break class invariants.
        throw new UnsupportedOperationException();
    }

    @Override
    public Iterator<E> iterator() {
        return new RingBufferBQIterator();
    }

    public static void main(String[] args) {
        // The shared buffer
        RingBufferBQ<Integer> b = new RingBufferBQ<>(1);

        // Task for producer threads to perform
        Runnable p = () -> {
            for (int i = 0; i < 10; ++i) {
                b.put(i);
            }
            System.out.println("Producer done");
        };

        // Task for consumer threads to perform
        Runnable c = () -> {
            int sum = 0;
            for (int i = 0; i < 10; ++i) {
                Integer j = b.get();
                sum += j;
            }
            System.out.println("Consumer done; sum: " + sum);
        };

        // Create and start producer and consumer threads
        int np = 5;  // Number of producer threads
        int nc = 5;  // Number of consumer threads
        for (int i = 0; i < np; ++i) {
            new Thread(p).start();
        }
        for (int i = 0; i < nc; ++i) {
            new Thread(c).start();
        }

        System.out.println("Main done");
    }

    /**
     * An iterator over a RingBufferBQ.
     */
    public class RingBufferBQIterator implements Iterator {

        /**
         * Number of entries in the RingBufferBQ that was iterated by this iterator.
         */
        int count;

        public RingBufferBQIterator() {
            count = 0;
        }

        /**
         * Returns {@code true} if the iteration has more elements. (In other words, returns
         * {@code true} if {@link #next} would return an element rather than throwing an exception.)
         *
         * @return {@code true} if the iteration has more elements
         */
        @Override
        public boolean hasNext() {
            return count < size;
        }

        /**
         * Returns the next element in the iteration.
         *
         * @return the next element in the iteration
         * @throws NoSuchElementException if the iteration has no more elements
         */
        @Override
        public E next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            E currElement = store[(iHead + count) % store.length];
            count++;
            return currElement;
        }
    }
}
