import java.io.IOException;
import java.io.BufferedReader;
import java.io.FileReader;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/** A labeled vertex in a graph that knows its outgoing edges. */
public class Vertex {
    /** This vertex's label. */
    private final String label;

    /** All vertices reachable via edges leaving this vertex. */
    private final Set<Vertex> successors;

    /** Create a new vertex with the specified label and no outgoing edges. */
    private Vertex(String label) {
        this.label = label;
        successors = new HashSet<>();
    }

    /** Return this vertex's label. */
    public String label() {
        return label;
    }

    /** Return an (immutable) collection supporting iteration over all successor vertices. */
    public Iterable<Vertex> successors() {
        return Collections.unmodifiableSet(successors);
    }

    /** Add an outgoing edge to vertex `n`. */
    private void addSuccessor(Vertex n) {
        successors.add(n);
    }

    /**
     * Create a graph (vertex set) based on vertices and edges listed in file `fileName`.
     * The graph is represented by adjacency lists (technically sets).
     * The file format is similar to GraphViz dot files.  A line must follow one of these patterns:
     * * Blank line (or only whitespace) - ignored
     * * Comment line starting with '#' - ignored
     * * Isolated vertex line: any label not containing {@code "->"}
     * * Edge line with format {@code srcLabel -> dstLabel}
     * Spaces at the beginning and end of vertex labels are trimmed.
     */
    public static Iterable<Vertex> readGraph(String fileName) throws IOException {
        Map<String, Vertex> index = new HashMap<>();
        try (BufferedReader in = new BufferedReader(new FileReader(fileName))) {
            for (String line = in.readLine(); line != null; line = in.readLine()) {
                // Skip blank lines and comment lines
                if (line.startsWith("#") || line.trim().isEmpty()) {
                    continue;
                }

                // Parse vertex labels from line
                String[] tokens = line.split("->");
                if (tokens.length > 2) {
                    throw new IOException("Invalid file format");
                }
                String srcLabel = tokens[0].trim();

                // Get first vertex from index, adding new vertex if necessary
                Vertex src = index.computeIfAbsent(srcLabel, k -> new Vertex(k));

                // Line defines an edge, not just a vertex
                if (tokens.length == 2) {
                    String dstLabel = tokens[1].trim();

                    // Get successor vertex from index, adding new vertex if necessary
                    Vertex dst = index.computeIfAbsent(dstLabel, k -> new Vertex(k));

                    // Add edge
                    src.addSuccessor(dst);
                }
            }
        }
        return index.values();
    }
}
