import java.io.IOException;
import java.io.BufferedReader;
import java.io.FileReader;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/** A labeled vertex in a graph that knows its outgoing edges. */
public class Node {
    /** This node's label. */
    private final String label;

    /** All nodes reachable via edges leaving this node. */
    private final Set<Node> successors;

    /** Create a new node with the specified label and no outgoing edges. */
    private Node(String label) {
        this.label = label;
        successors = new HashSet<>();
    }

    /** Return this node's label. */
    public String label() {
        return label;
    }

    /** Return an (immutable) collection supporting iteration over all successor nodes. */
    public Iterable<Node> successors() {
        return Collections.unmodifiableSet(successors);
    }

    /** Add an outgoing edge to node `n`. */
    private void addSuccessor(Node n) {
        successors.add(n);
    }

    /**
     * Create a graph (vertex set) based on vertices and edges listed in file `fileName`.
     * The graph is represented by adjacency lists (technically sets).
     * The file format is similar to GraphViz dot files.  A line must follow one of these patterns:
     * * Blank line (or only whitespace) - ignored
     * * Comment line starting with '#' - ignored
     * * Isolated node line: any label not containing {@code "->"}
     * * Edge line with format {@code srcLabel -> dstLabel}
     * Spaces at the beginning and end of node labels are trimmed.
     */
    public static Iterable<Node> readGraph(String fileName) throws IOException {
        Map<String, Node> index = new HashMap<>();
        try (BufferedReader in = new BufferedReader(new FileReader(fileName))) {
            for (String line = in.readLine(); line != null; line = in.readLine()) {
                // Skip blank lines and comment lines
                if (line.startsWith("#") || line.trim().isEmpty()) {
                    continue;
                }

                // Parse node labels from line
                String[] tokens = line.split("->");
                if (tokens.length > 2) {
                    throw new IOException("Invalid file format");
                }
                String srcLabel = tokens[0].trim();

                // Get first node from index, adding new node if necessary
                Node src = index.computeIfAbsent(srcLabel, k -> new Node(k));

                // Line defines an edge, not just a node
                if (tokens.length == 2) {
                    String dstLabel = tokens[1].trim();

                    // Get successor node from index, adding new node if necessary
                    Node dst = index.computeIfAbsent(dstLabel, k -> new Node(k));

                    // Add edge
                    src.addSuccessor(dst);
                }
            }
        }
        return index.values();
    }
}
