import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.JComponent;

/**
 * A WorldView is a custom Swing component that renders a "world" that looks like graph paper with a
 * little person (the "walker") who can be walked around using the arrow keys.
 */
public class WorldView extends JComponent {

    /**
     * Number of pixels between grid lines (inclusive).
     */
    private static final int CELL_SIZE = 30;

    // model state - ideally, would refactor into separate GUI-independent
    // code as part of MVC pattern.
    /**
     * Horizontal walker position (number of cells to the right of the left boundary).
     * Non-negative.
     */
    int walkX;

    /**
     * Vertical walker position (number of cells below the top boundary).  Non-negative.
     */
    int walkY;

    /**
     * Name of walker.
     */
    String name;

    /**
     * Create a new WorldView with the walker in the upper-left corner.
     */
    public WorldView() {
        // Initialize model state
        walkX = 0;
        walkY = 0;
        name = "";

        setFocusable(true); // allow keyboard events
        setPreferredSize(new Dimension(300, 300));

        // Use anonymous subclass of MouseAdapter to listen for mouse events.
        addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                requestFocus(); // get keyboard focus when clicked
            }
        });

        // Use anonymous subclass of KeyAdapter to listen for keyboard events.
        addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                switch (e.getKeyCode()) {
                    case KeyEvent.VK_LEFT:
                        moveXY(-1, 0);
                        break;
                    case KeyEvent.VK_RIGHT:
                        moveXY(1, 0);
                        break;
                    case KeyEvent.VK_UP:
                        moveXY(0, -1);
                        break;
                    case KeyEvent.VK_DOWN:
                        moveXY(0, 1);
                        break;
                    // TODO 5: Add another key that does something of your choice.
                }
            }
        });

        // Use anonymous subclass of ComponentAdapter to keep walker within bounds after resizing.
        addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                moveXY(0, 0);
            }
        });
    }

    /**
     * WorldView defines its own custom paintComponent() method, which you can modify.
     */
    @Override
    public void paintComponent(Graphics g) {
        // Draw background grid
        int w = getWidth();
        int h = getHeight();
        g.setColor(Color.getHSBColor(0.5f, 0.3f, 0.9f));
        g.clearRect(0, 0, w, h);
        for (int x = 0; x < w; x += CELL_SIZE) {
            g.drawLine(x, 0, x, h);
        }
        for (int y = 0; y < h; y += CELL_SIZE) {
            g.drawLine(0, y, w, y);
        }

        // Compute location of center of walker in pixels
        int cx = walkX * CELL_SIZE + CELL_SIZE / 2;
        int cy = walkY * CELL_SIZE + CELL_SIZE / 2;

        // Draw walker stick figure
        final int s = CELL_SIZE;  // Alias to keep math concise
        g.setColor(Color.BLACK);
        g.drawArc(cx - s / 6, cy - s / 3, s / 3, s / 3, 0, 360); // head
        g.drawLine(cx, cy, cx, cy + s / 4);                      // body
        g.drawLine(cx, cy + s / 4, cx + s / 4, cy + s / 2);      // right leg
        g.drawLine(cx, cy + s / 4, cx - s / 4, cy + s / 2);      // left leg
        // TODO 4: give the walker arms

        // Write walker's name
        g.setColor(Color.BLUE);
        g.drawString(name, cx - s / 2, cy + s / 2);
    }

    // model operations

    /**
     * Set the name of the walker, which is displayed with them.
     */
    public void setWalkerName(String name) {
        this.name = name;
        repaint();
    }

    /**
     * Move the walker by (dx, dy) grid positions, but keep them in bounds.
     */
    public void moveXY(int dx, int dy) {
        walkX += dx;
        walkY += dy;
        if (walkX < 0) {
            walkX = 0;
        }
        if (walkY < 0) {
            walkY = 0;
        }
        if ((walkX + 1) * CELL_SIZE >= getWidth()) {
            walkX = getWidth() / CELL_SIZE - 1;
        }
        if ((walkY + 1) * CELL_SIZE >= getHeight()) {
            walkY = getHeight() / CELL_SIZE - 1;
        }
        repaint();
    }
}
